<?php
/**
 * @author    Oliver Schieche <lispian@schieche.email>
 * @copyright 2018 Oliver Schieche
 */
namespace Ghoti\Tools\Lispian\VM\Runner;

use Ghoti\Tools\Lispian\Exception\VM\RunnerException;
use Ghoti\Tools\Lispian\Exception\VM\SeriousBugException;
use Ghoti\Tools\Lispian\Exception\VM\UnknownSymbolException;
use Ghoti\Tools\Lispian\VM\BytecodeSerializer;
use Ghoti\Tools\Lispian\VM\OpCode;
use Ghoti\Tools\Lispian\VM\Symbol\TrackedSymbol;

/**
 * Class BytecodeRunner
 *
 * @package Ghoti\Tools\Lispian\VM\Runner
 */
class BytecodeRunner extends AbstractRunner
{
    /** @var mixed A register */
    protected $eax;
    /** @var int Instruction pointer */
    protected $eip;
    /** @var array */
    protected $instructions;
    /** @var array */
    protected $scope;
    /** @var array */
    protected $stack;

    /**
     * @param string $bytecode
     * @return mixed
     * @throws RunnerException
     * @throws SeriousBugException
     */
    public function run(string $bytecode)
    {
        $serializer = new BytecodeSerializer();
        $this->instructions = $serializer->unserialize($bytecode);
        $this->eip = 0;
        $this->stack = [];

        $totalInstructions = \count($this->instructions);
        while ($this->eip < $totalInstructions) {
            $instruction = $this->popInstruction();

            switch ($instruction)
            {
                case OpCode::ADD:
                    $this->eax += $this->popStack();
                    break;

                case OpCode::CALL:
                    $this->executeCall();
                    break;

                case OpCode::CMPEQ:
                    $left = $this->popStack();
                    $this->eax = +($left === $this->eax);
                    break;

                case OpCode::CMPNE:
                    $left = $this->popStack();
                    $this->eax = +($left !== $this->eax);
                    break;

                case OpCode::CMPGT:
                    $left = $this->popStack();
                    $this->eax = +($left > $this->eax);
                    break;

                case OpCode::CMPGTE:
                    $left = $this->popStack();
                    $this->eax = +($left >= $this->eax);
                    break;

                case OpCode::CMPLT:
                    $left = $this->popStack();
                    $this->eax = +($left < $this->eax);
                    break;

                case OpCode::CMPLTE:
                    $left = $this->popStack();
                    $this->eax = +($left <= $this->eax);
                    break;

                case OpCode::DIV:
                    if (0 === $this->eax) {
                        throw new RunnerException('Division by zero.', 1541322566);
                    }

                    $this->eax = $this->popStack() / $this->eax;
                    break;

                case OpCode::DOT:
                    $this->executeDot();
                    break;

                case OpCode::DOTCALL:
                    $this->executeDotCall();
                    break;

                case OpCode::DUMP:
                    $this->dump();
                    break;

                case OpCode::INVOKE:
                    $this->executeInvoke();
                    break;

                case OpCode::JMP:
                    $numInstructions = $this->popInstruction();
                    $this->eip += $numInstructions;
                    break;

                case OpCode::JNZ:
                    $frames = $this->popInstruction();
                    if ($this->eax) {
                        $this->jump($frames - 2);
                    }
                    break;

                case OpCode::JZ:
                    $frames = $this->popInstruction();
                    if (!$this->eax) {
                        $this->jump($frames - 2);
                    }
                    break;

                case OpCode::LOAD:
                    $index = $this->popInstruction();
                    $this->eax = $this->instructions[$index];
                    break;

                case OpCode::LOOKUP:
                    $symbol = $this->popInstruction();
                    $this->eax = $this->lookupSymbol($symbol);
                    if ($this->eax instanceof TrackedSymbol) {
                        $this->eax = $this->eax->getValue();
                    }
                    break;

                case OpCode::MOVA:
                    $this->eax = $this->popInstruction();
                    break;

                case OpCode::MUL:
                    $this->eax *= $this->popStack();
                    break;

                case OpCode::NOT:
                    $this->eax = +!$this->eax;
                    break;

                case OpCode::NOP:
                    break;

                case OpCode::POPSYM:
                    $this->executePopSymbol();
                    break;

                case OpCode::POPVAR:
                    $this->executePopVar();
                    break;

                case OpCode::POW:
                    $base = $this->popStack();
                    $this->eax = $base ** $this->eax;
                    break;

                case OpCode::PR0GRAM_INIT:
                    $this->eax = 0xCCCCCCCC;
                    $this->scope = [];
                    $this->stack = [];
                    break;

                case OpCode::PR0GRAM_END:
                    return $this->endProgram();

                case OpCode::POP:
                    $this->eax = $this->popStack();
                    break;

                case OpCode::PUSH:
                    $this->pushStack($this->eax);
                    break;

                case OpCode::RET:
                    $this->eip = $this->popStack();
                    break;

                case OpCode::SCOPE_RET:
                    \array_shift($this->scope);
                    break;

                case OpCode::SCOPE_INIT:
                    \array_unshift($this->scope, ['VARS' => []]);
                    break;

                case OpCode::SUB:
                    $this->eax = $this->popStack() - $this->eax;
                    break;

                default:
                    --$this->eip;
                    $this->dump($instruction);
                    throw new RunnerException(\sprintf('Unimplemented instruction code "%s" at EIP %d', $instruction, $this->eip));
            }
        }

        throw new SeriousBugException('EIP exceeded number of instructions; report this as a bug.');
    }

    /**
     * @param mixed ...$extraDump
     */
    protected function dump(...$extraDump)
    {
        $escape = function($value) {
            if (\is_callable($value)) {
                return '(callable)';
            }

            if (\is_array($value)) {
                return \json_encode($value);
            }

            if (\is_int($value) || \is_float($value)) {
                return $value;
            }

            if (\is_object($value)) {
                return \get_class($value);
            }

            return \preg_replace_callback('~[\x00-\x1f]~', function ($match) {
                static $replacements = [
                    "\e" => '\\e',
                    "\f" => '\\f',
                    "\n" => '\\n',
                    "\r" => '\\r',
                    "\t" => '\\t',
                    "\v" => '\\v',
                ];

                return $replacements[$match[0]] ?? \sprintf('\\x%02x', \ord($match[0]));
            }, $value);
        };

        printf("%s\n", \str_repeat('-', 40));
        printf("EIP: %d\n", $this->eip);
        printf("EAX: %s\n", $escape($this->eax));
        printf("%s\n", \str_repeat('-', 40));
        printf("STACK:\n");
        foreach (\array_reverse($this->stack) as $index => $item) {
            printf("%s  [%s]\n", $index ? '  ' : '->', $escape($item));
        }
        printf("%s\n", \str_repeat('-', 40));

        foreach ($this->instructions as $frame => $instruction) {
            $extra = '';
            if (\in_array($instruction, [OpCode::JNZ, OpCode::JZ, OpCode::JMP], true)) {
                $extra = \sprintf(' --> %d', $frame + $this->instructions[$frame + 1]);
            } elseif (OpCode::LOAD === $instruction) {
                $extra = \sprintf(' --> <%s>', $escape($this->instructions[$this->instructions[$frame + 1]]));
            }
            \printf("%s %3d %s%s\n", $this->eip === $frame ? '->' : '  ', $frame,
                \is_string($instruction) ? $escape($instruction) : \sprintf('%02x (%d)', $instruction, $instruction), $extra);
        }

        if (\count($extraDump)) {
            \var_dump(...$extraDump);
        }
    }

    /**
     * @return mixed
     */
    protected function endProgram()
    {
        if (0 !== \count($this->stack)) {
            $this->dump();
            \trigger_error('Unclean program termination.', \E_USER_ERROR);
        }

        return $this->eax;
    }

    /**
     * @return mixed
     */
    protected function peekInstruction()
    {
        return $this->instructions[$this->eip];
    }

    /**
     * @return mixed
     */
    protected function popInstruction()
    {
        return $this->instructions[$this->eip++];
    }

    /**
     * @return mixed
     * @throws RunnerException
     */
    protected function popStack()
    {
        if (!\count($this->stack)) {
            throw new RunnerException('Stack underflow');
        }

        return \array_pop($this->stack);
    }

    /**
     * @param mixed ...$values
     * @return BytecodeRunner
     */
    protected function pushStack(...$values): self
    {
        \array_push($this->stack, ...$values);
        return $this;
    }

    /**
     * @param string $symbol
     * @return mixed
     * @throws RunnerException
     */
    protected function lookupSymbol(string $symbol)
    {
        foreach ($this->scope as $scope) {
            if (\array_key_exists($symbol, $scope['VARS'])) {
                return $scope['VARS'][$symbol];
            }
        }

        return $this->lookupInProvider($symbol);
    }

    /**
     * @param string $symbol
     * @return mixed
     * @throws RunnerException
     */
    protected function lookupInProvider(string $symbol)
    {
        try {
            return $this->symbolProvider->lookup($symbol);
        } catch (UnknownSymbolException $exception) {
            throw new RunnerException($exception->getMessage(), $exception->getCode(), $exception);
        }
    }

    /**
     * @param $frames
     * @return BytecodeRunner
     */
    protected function jump($frames): self
    {
        $this->eip += $frames;
        return $this;
    }

    /**
     * @return BytecodeRunner
     */
    protected function executeCall(): self
    {
        $target = $this->popInstruction();
        $this->pushStack($this->eip);
        $this->eip = $target;

        return $this;
    }

    /**
     * @return BytecodeRunner
     * @throws RunnerException
     */
    protected function executeDot(): self
    {
        $member = $this->eax;
        $value = $this->popStack();

        if (\is_array($value)) {
            return $this->executeDotArray($value, $member);
        }

        if (\is_object($value)) {
            return $this->executeDotObject($value, $member);
        }

        // This stays until we need to refine it...
        die(\sprintf('Cannot dot-access this: %s', \print_r($value, true)));
    }

    /**
     * @param array $value
     * @param $member
     * @return $this
     */
    protected function executeDotArray(array $value, $member): self
    {
        if (\is_int($member) && $member < 0) {
            $member += \count($value);
        }

        if (!\array_key_exists($member, $value)) {
            \trigger_error(\sprintf('Undefined index "%s"', $member), \E_USER_WARNING);
            $this->eax = null;
            return $this;
        }

        $this->eax = $value[$member];
        return $this;
    }

    /**
     * @return BytecodeRunner
     * @throws RunnerException
     */
    protected function executeDotCall(): self
    {
        $parameters = [];
        $numParameters = $p = $this->popStack();
        while ($p--) {
            \array_unshift($parameters, $this->popStack());
        }
        $value = $this->popStack();
        $member = $this->eax;

        try {
            $reflection = new \ReflectionObject($value);
            $method = $reflection->getMethod($member);
        } /** @noinspection PhpRedundantCatchClauseInspection */ catch(\ReflectionException $exception) {
            throw new RunnerException(\sprintf('While attempting to call %s::%s(): %s: %s',
                \get_class($value), $member, \get_class($exception), $exception->getMessage()), 1541335718);
        }

        try {
            if (!$method->isPublic()) {
                throw new RunnerException('attempted to access non-public method', 1541331233);
            }

            $requiredParameters = $method->getNumberOfRequiredParameters();
            if ($numParameters < $requiredParameters) {
                throw new RunnerException(\sprintf('method requires at least %d parameter(s); got %d instead', $requiredParameters, $numParameters), 1541335871);
            }

            $requiredParameters = $method->getNumberOfParameters();
            if ($numParameters > $requiredParameters) {
                throw new RunnerException(\sprintf('method accepts %d parameters at most; got %d instead', $requiredParameters, $numParameters), 1541335971); // time code not a coincidence
            }

            $this->eax = $method->invokeArgs($value, $parameters);
            return $this;

        } catch (RunnerException $exception) {
            throw new RunnerException(\sprintf('While attempting to call %s::%s(): %s: %s',
                $reflection->getName(), $method->getName(), \get_class($exception), $exception->getMessage()),
                $exception->getCode(), $exception);
        }
    }

    /**
     * @param $value
     * @param $member
     * @return $this
     * @throws RunnerException
     */
    protected function executeDotObject($value, $member): self
    {
        if (\property_exists($value, $member)) {
            $this->eax = $value->$member;
            return $this;
        }

        \trigger_error(\sprintf('Undefined property %s#%s', \get_class($value), $member), \E_USER_WARNING);
        $this->eax = null;
        return $this;
    }

    /**
     * @return BytecodeRunner
     * @throws RunnerException
     */
    protected function executeInvoke(): self
    {
        $numParameters = $this->popInstruction();
        $methodName = $this->popInstruction();
        $symbol = $this->lookupInProvider($methodName);

        if (!\is_callable($symbol)) {
            throw new RunnerException(\sprintf('Symbol "%s" is not callable', $methodName), 1541265104);
        }

        $parameters = [];
        while ($numParameters--) {
            \array_unshift($parameters, $this->popStack());
        }

        $this->eax = $symbol(...$parameters);

        return $this;
    }

    /**
     * @return BytecodeRunner
     * @throws RunnerException
     */
    protected function executePopSymbol(): self
    {
        $symbolName = $this->peekInstruction();

        try {
            $symbol = $this->lookupSymbol($symbolName);

            if (!$symbol instanceof TrackedSymbol) {
                return $this->executePopVar();
            }

            $this->popInstruction();
            $value = $this->popStack();
            $symbol->setValue($value);

            $this->eax = $value;
        } catch (RunnerException $e) {
            return $this->executePopVar();
        }

        return $this;
    }

    /**
     * This method MUST NOT modify EAX!
     *
     * @return BytecodeRunner
     * @throws RunnerException
     */
    protected function executePopVar(): self
    {
        $symbol = $this->popInstruction();
        $this->scope[0]['VARS'][$symbol] = $this->popStack();
        return $this;
    }
}
