<?php
/**
 * @author    Oliver Schieche <lispian@schieche.email>
 * @copyright 2018 Oliver Schieche
 */

namespace Test\Ghoti\Tools\Lispian\Unit\Lexer;

use Ghoti\Tools\Lispian\Exception\Lexer\AbstractLexerException;
use Ghoti\Tools\Lispian\Exception\Lexer\InvalidCharacterException;
use Ghoti\Tools\Lispian\Exception\Lexer\UnterminatedRegexException;
use Ghoti\Tools\Lispian\Exception\Lexer\UnterminatedStringException;
use Ghoti\Tools\Lispian\Lexical\Lexer;
use PHPUnit\Framework\TestCase;
use Test\Ghoti\Tools\Lispian\Parser\NullParser;

use function get_class, sprintf;

class LexerTest extends TestCase
{
    /**
     * @dataProvider lexerTestInput
     * @param string $source
     * @param array $expectToken
     * @param int|null $expectLine
     * @param int|null $expectColumn
     * @throws InvalidCharacterException
     * @throws UnterminatedStringException
     * @throws UnterminatedRegexException
     */
    public function testLex(string $source, array $expectToken, int $expectLine = null, int $expectColumn = null)
    {
        $parser = new NullParser();
        $lexer = new Lexer($source);

        foreach ($expectToken as $expectedToken) {
            $token = $lexer->lex($parser);
            $this->assertEquals($expectedToken, $token);
        }

        if (null !== $expectLine) {
            $this->assertEquals($expectLine, $lexer->getLine(), 'Line mismatch');
        }
        if (null !== $expectColumn) {
            $this->assertEquals($expectColumn, $lexer->getColumn(), 'Column mismatch');
        }
    }

    /**
     * @dataProvider lexerTestInvalid
     * @param string $source
     * @param string $expectedExceptionClass
     * @param string $expectedErrorMessage
     */
    public function testInvalidTokens(string $source, string $expectedExceptionClass, string $expectedErrorMessage)
    {
        $parser = new NullParser();
        $lexer = new Lexer($source);

        do {
            try {
                $token = $lexer->lex($parser);
            } catch(AbstractLexerException $exception) {
                $this->assertEquals($expectedExceptionClass, get_class($exception));
                $this->assertEquals($expectedErrorMessage, $exception->getMessage());
                return;
            }

            if ('' === $token[0] && null === $token[1]) {
                break;
            }
        } while (true);

        $this->fail(sprintf('Source "%s" should have failed with "%s" exception, message "%s" but did not.',
            $source, $expectedExceptionClass, $expectedErrorMessage));
    }

    /**
     * @return array
     */
    public function lexerTestInput(): array
    {
        return [
            ["    \n   \t   ", [['', null]], 2, 8],
            ['$a', [['T_VARIABLE', 'a']], 1, 3],
            ['$an_Excessively_L0NG_variable_name_001', [['T_VARIABLE', 'an_Excessively_L0NG_variable_name_001']], 1, 39],
            ['true', [['T_CONSTANT', 'true']], 1, 5],
            ['false', [['T_CONSTANT', 'false']], 1, 6],
            ['null', [['T_CONSTANT', 'null']], 1, 5],

            ['gork', [['T_IDENTIFIER', 'gork']], 1, 5],
            ["\tsmurch", [['T_IDENTIFIER', 'smurch']], 1, 8],
            ['1337', [['T_INTEGER', 1337]], 1, 5],
            ['-1337', [['T_INTEGER', -1337]], 1, 6],
            ['13.37', [['T_NUMBER', 13.37]], 1, 6],
            ['-13.37', [['T_NUMBER', -13.37]], 1, 7],

            ['or', [['or', 'or']], 1, 3],
            ['and', [['and', 'and']], 1, 4],
            ['not', [['not', 'not']], 1, 4],

            ['inc', [['T_POST_INCREMENT', 'inc']], 1, 4],
            ['dec', [['T_POST_INCREMENT', 'dec']], 1, 4],

            ['defun', [['T_DEFUN', 'defun']], 1, 6],
            ['do', [['T_DO', 'do']], 1, 3],
            ['if', [['T_IF', 'if']], 1, 3],
            ['for', [['T_FOR', 'for']], 1, 4],
            ['let', [['T_LET', 'let']], 1, 4],

            ['+1', [['+','+']], 1, 2],
            ['-Z', [['-','-']], 1, 2],
            ['*3', [['*','*']], 1, 2],
            ['/4', [['/','/']], 1, 2],

            ['"Hello\t\"world.\"\n"', [['T_STRING',"Hello\t\"world.\"\n"]], 1, 22],

            // Assert pre-increment comes before addition
            ['+++', [['T_PRE_INCREMENT', '++'], ['+', '+']], 1, 4],
            // Assert pre-decrement comes before subtraction
            ['---', [['T_PRE_INCREMENT', '--'], ['-', '-']], 1, 4],
            // Assert exponentiation comes before multiplication
            ['***', [['**', '**'], ['*', '*']], 1, 4],
            // Assert invocation parameter operator comes before subtraction
            ['->-', [['->', '->'], ['-', '-']], 1, 4]
        ];
    }

    /**
     * @return array
     */
    public function lexerTestInvalid(): array
    {
        return [
            ['$-a', InvalidCharacterException::class, 'Unexpected input at line 1, column 1: "$-a"'],
            ["# Some long comment\n(eq \$a \$b)@", InvalidCharacterException::class, 'Unexpected input at line 2, column 11: "@"'],
            ['"This is a string that never ends', UnterminatedStringException::class, 'Unterminated string at line 1, column 1: ""This is"']
        ];
    }
}
