<?php
/**
 * @author    Oliver Schieche <lispian@schieche.email>
 * @copyright 2018 Oliver Schieche
 */
namespace Ghoti\Tools\Lispian\VM\Support\Builtins\Functions;

use Ghoti\Tools\Lispian\Exception\VM\FunctionExecutionException;
use Ghoti\Tools\Lispian\Exception\VM\PhpErrorException;
use Ghoti\Tools\Lispian\VM\Support\SymbolProvider;
use Exception;

use const E_ALL, STDOUT;
use function fwrite, is_int, set_error_handler, sprintf;

/**
 * Class BuiltinFunctionsFactory
 *
 * @package Ghoti\Tools\Lispian\VM\Support\Builtins\Functions
 */
class BuiltinFunctionsFactory
{
    /** @var string[] */
    const BUILTINS = [
        'array_push' => 'push',
        'count',
        'date',
        'printf',
        'strftime',
        'sprintf',
        'time',
        'var_dump' => 'dump'
    ];
    /** @var BuiltinFunctionsFactory */
    static protected $instance;

    /**
     * @return BuiltinFunctionsFactory
     */
    public static function getInstance(): self
    {
        if (null === static::$instance) {
            static::$instance = new static();
        }

        return static::$instance;
    }

    /**
     * @param SymbolProvider $provider
     */
    public function addToProvider(SymbolProvider $provider)
    {
        $print = static function(...$arguments) {
            foreach($arguments as $argument) {
                fwrite(STDOUT, $argument);
            }
        };

        $builtins = [];

        $builtins['print'] = static function(...$arguments) use($print) {
            $print(...$arguments);
            return 1;
        };

        $builtins['println'] = static function(...$arguments) use($print) {
            $arguments[] = "\n";
            $print(...$arguments);
            return 1;
        };

        foreach (self::BUILTINS as $function => $alias) {
            if (is_int($function)) {
                $function = $alias;
            }
            $builtins[$alias] = $this->createBuiltinAlias($function);
        }

        $provider->setSymbols($builtins);
    }

    /**
     * @param string $hostFunction
     * @return callable
     */
    protected function createBuiltinAlias(string $hostFunction): callable
    {
        return static function(...$arguments) use($hostFunction) {
            try {
                $errorHandler = set_error_handler(static function($errno, $errstr, $errfile, $errline) {
                    throw new PhpErrorException($errno, $errstr, $errfile, $errline);
                }, E_ALL);

                try {
                    return $hostFunction(...$arguments);
                } finally {
                    set_error_handler($errorHandler);
                }
            } catch(Exception $exception) {
                throw new FunctionExecutionException(sprintf('Error in host function execution of "%s": %s',
                    $hostFunction, $exception->getMessage()), $exception->getCode(), $exception);
            }
        };
    }
}
