<?php /** @noinspection PhpUndefinedFieldInspection */

/**
 * @author    Oliver Schieche <lispian@schieche.email>
 * @copyright 2019 Oliver Schieche
 */
namespace Ghoti\Tools\Lispian\VM\Runner\BytecodeHelper;

use Ghoti\Tools\Lispian\VM\OpCode;

use function array_key_exists, array_reverse, count,
    get_class, in_array, is_array, is_callable, is_float, is_int, is_object, is_string, json_encode, ord,
    preg_replace_callback, printf, sprintf, str_repeat, var_dump;

/**
 * Class CodeDumperTrait
 * @package Ghoti\Tools\Lispian\VM\Runner\BytecodeHelper
 */
trait CodeDumperTrait
{
    protected function dump(...$extraDump)
    {
        $escape = static function($value) {
            if (is_callable($value)) {
                if (is_string($value)) {
                    return "(callable) --> \e[1;4;31m$value()\e[0m";
                }
                return '(callable)';
            }

            if (is_array($value)) {
                return json_encode($value);
            }

            if (is_int($value) || is_float($value)) {
                return $value;
            }

            if (is_object($value)) {
                return get_class($value);
            }

            return preg_replace_callback('~[\x00-\x1f]~', static function($match) {
                static $replacements = [
                    "\e" => '\\e',
                    "\f" => '\\f',
                    "\n" => '\\n',
                    "\r" => '\\r',
                    "\t" => '\\t',
                    "\v" => '\\v',
                ];

                return $replacements[$match[0]] ?? sprintf('\\x%02x', ord($match[0]));
            }, $value);
        };

        $colorize = static function(string $instruction) {
            static $colors = [
                'jmp' => '1;33',
                'jnz' => '1;33',
                'jz'  => '1;33',
                'load' => '1;34',
                'loada' => '1;34',
                'lookup' => '1;34',
                'invoke' => '1;35',

                'spush' => '1;36',

            ];

            if (!array_key_exists($instruction, $colors)) {
                return $instruction;
            }

            return sprintf("\e[%sm%s\e[0m", $colors[$instruction], $instruction);
        };

        printf("%s\n", str_repeat('-', 40));
        printf("EIP: %d\n", $this->eip);
        printf("EAX: %s\n", $escape($this->eax));
        printf("%s\n", str_repeat('-', 40));
        printf("STACK:\n");
        foreach (array_reverse($this->stack) as $index => $item) {
            printf("%s  [%s]\n", $index ? '  ' : '->', $escape($item));
        }
        printf("%s\n", str_repeat('-', 40));

        $inData = false;
        $dataEnd = null;

        foreach ($this->instructions as $frame => $instruction) {
            if (3 === $frame && $this->instructions[1] === OpCode::get(OpCode::JMP)) {
                $inData = true;
                print("\e[1;30m----vv DATA SEGMENT vv------------------\e[0m\n");
                $dataEnd = $frame + $this->instructions[$frame - 1];
            } elseif ($dataEnd === $frame) {
                $inData = false;
                print("\e[1;30m----^^ DATA SEGMENT ^^------------------\e[0m\n");
            }

            $extra = '';
            if (in_array($instruction, [OpCode::get(OpCode::JNZ), OpCode::get(OpCode::JZ), OpCode::get(OpCode::JMP)], true)) {
                $extra = sprintf(' --> %d', $frame + $this->instructions[$frame + 1] + 2); // +2 cause EIP did not move
            } elseif (OpCode::get(OpCode::LOAD) === $instruction) {
                $extra = sprintf(' --> >%s<', $escape($this->instructions[$this->instructions[$frame + 1]]));
            }

            printf("%s \e[1;30m%3d\e[0m %s%s\n", $this->eip === $frame ? '->' : '  ', $frame,
                $inData ? ("\e[1;30m" . (is_string($instruction) ? $escape($instruction) : sprintf('0x%02x (%d)', $instruction, $instruction)) . "\e[0m") :
                    (is_string($instruction) ? $colorize($escape($instruction)) : sprintf('0x%02x (%d)', $instruction, $instruction)), $extra);
        }

        printf("%s\n", str_repeat('-', 40));

        if (count($extraDump)) {
            var_dump(...$extraDump);
            printf("%s\n", str_repeat('-', 40));
        }
    }
}
