# Lispian documentation #

## External and tracked variables ##

In order to advance the "hello world" stage and actually interact with a 
calling program, you'll have to provide some context to a *Lispian* runner.

This chapter covers how external data can be passed to a script and modified by
a script.

### Preamble ###

From [the most basic way to execute a script](../intro/getting-started.md) we've
learned that the byte-code runner requires a "provider".

Providers are the way to inject data from the running PHP script's scope into a
*Lispian* script. They act like a _super-scope_ and may provide pre-initialized
variables for the underlying *Lispian* script.

### Providing external variables ###

In order to inject variables or functions into *Lispian* context, this library
provides both `SymbolProvider` and `BuiltinsProvider`. Both reside in the
`Ghoti\Tools\Lispian\VM\Support` namespace. The former is empty and practically 
does nothing while the latter extends from the former and provides for 
[built-in functions](../features/builtin-functions.md).

Both providers' constructors accept an array of variables or functions that are
made available to a *Lispian* script.

```php
use Ghoti\Tools\Lispian\Parser;
use Ghoti\Tools\Lispian\VM\Runner\BytecodeRunner;
use Ghoti\Tools\Lispian\VM\Support\BuiltinsProvider;

$source = <<<EOS
    (write $variable)
EOS;

// Initialize parser and let it do its thing to create a runnable node
$parser = new Parser($source);
$node = $parser->YYParse();

// Convert node to runnable byte-code
$byteCode = $node->toByteCode();

// Initialize a scope provider
$provider = new BuiltinsProvider([
    'variable' => 'Hello world.',
    'write' => function($what) {
        echo $what;
    }
]);

// Run compiled byte-code
$runner = new BytecodeRunner($provider);
$result = $runner->run($byteCode);
```

The above example illustrates how to provide both user-defined functions
and user-defined variables for *Lispian* to execute or evaluate.

### Tracked variables ###

While the byte-code runner's `run()` method returns the last *Lispian* 
expression's value, it may at times be necessary to modify more than one
value at once. In order to facilitate this, the library provides 
_variable tracking_ via the `Ghoti\Tools\Lispian\VM\Symbol\TrackedSymbol` 
class. 

The example to demonstrate this feature is somewhat abbreviated in order to
focus on the important parts:

```php
$var = 'Initial value';

$provider = new BuiltinsProvider([
    'external' => new Ghoti\Tools\Lispian\VM\Symbol\TrackedSymbol($var)
]);

$result = $runner->run($byteCode);

print "$var\n";
```

Assuming `$byteCode` is the following script, the symbol `$external` 
was used in the *Lispian* script and would have been modified:

```lisp
(print $external "\n")
(let $external "Some other value")
```
