<?php
/**
 * @author    Oliver Schieche <lispian@schieche.email>
 * @copyright 2018 Oliver Schieche
 */
namespace Ghoti\Tools\Lispian\VM\Runner;

use Ghoti\Tools\Lispian\Exception\VM\RunnerException;
use Ghoti\Tools\Lispian\Exception\VM\SeriousBugException;
use Ghoti\Tools\Lispian\Exception\VM\UnknownSymbolException;
use Ghoti\Tools\Lispian\VM\BytecodeSerializer;
use Ghoti\Tools\Lispian\VM\OpCode;
use Ghoti\Tools\Lispian\VM\Runner\BytecodeHelper\CodeDumperTrait;
use Ghoti\Tools\Lispian\VM\Runner\BytecodeHelper\RegexMatch;
use Ghoti\Tools\Lispian\VM\Symbol\TrackedSymbol;
use ReflectionException;
use ReflectionObject;
use Throwable;

use const E_USER_ERROR;
use const E_USER_WARNING;
use function array_key_exists, array_pop, array_push, array_shift, array_unshift, count, get_class, gettype, in_array,
    is_array, is_callable, is_int, is_object, is_string, print_r, property_exists, sprintf, strpos, trigger_error;

/**
 * Class BytecodeRunner
 *
 * @package Ghoti\Tools\Lispian\VM\Runner
 */
class BytecodeRunner extends AbstractRunner
{
    use CodeDumperTrait;

    /** @var mixed A register */
    protected $eax;
    /** @var int Instruction pointer */
    protected $eip;
    /** @var array */
    protected $instructions;
    /** @var array */
    protected $scope;
    /** @var array */
    protected $stack;

    /**
     * @param string $bytecode
     * @return mixed
     * @throws RunnerException
     * @throws SeriousBugException
     */
    public function run(string $bytecode)
    {
        $done = false;
        $mnemonicExecutors = [
            OpCode::ADD => function() {
                $this->eax += $this->popStack();
            },

            OpCode::CALL => function() {
                $this->executeCall();
            },

            OpCode::CMPIN => function() {
                $left = $this->popStack();

                if (is_string($left) && is_string($this->eax)) {
                    $this->eax = +(false !== strpos($this->eax, $left));
                } else {
                    $this->eax = +(is_array($this->eax) && in_array($left, $this->eax, true));
                }
            },
            OpCode::CMPEQ => function() {
                $left = $this->popStack();
                $this->eax = +($left === $this->eax);
            },
            OpCode::CMPNE => function() {
                $left = $this->popStack();
                $this->eax = +($left !== $this->eax);
            },
            OpCode::CMPGT => function() {
                $left = $this->popStack();
                $this->eax = +($left > $this->eax);
            },
            OpCode::CMPGTE => function() {
                $left = $this->popStack();
                $this->eax = +($left >= $this->eax);
            },
            OpCode::CMPLT => function() {
                $left = $this->popStack();
                $this->eax = +($left < $this->eax);
            },
            OpCode::CMPLTE => function() {
                $left = $this->popStack();
                $this->eax = +($left <= $this->eax);
            },

            OpCode::DIV => function() {
                if (0 === $this->eax) {
                    throw new RunnerException('Division by zero.', 1541322566);
                }

                $this->eax = $this->popStack() / $this->eax;
            },

            OpCode::DOT => function() {
                $this->executeDot();
            },
            OpCode::DOTCALL => function() {
                $this->executeDotCall();
            },

            OpCode::DUMP => function() {
                $this->dump();
            },

            OpCode::INVOKE => function() {
                $this->executeInvoke();
            },

            OpCode::JMP => function() {
                $this->jump($this->popInstruction());
            },
            OpCode::JNZ => function() {
                $frames = $this->popInstruction();
                if ($this->eax) {
                    $this->jump($frames);
                }
            },
            OpCode::JZ => function() {
                $frames = $this->popInstruction();
                if (!$this->eax) {
                    $this->jump($frames);
                }
            },

            OpCode::LOAD => function() {
                $index = $this->popInstruction();
                $this->eax = $this->instructions[$index];
            },
            OpCode::LOADA => function() {
                $index = $this->popInstruction();
                $count = $this->instructions[$index++];
                $array = [];
                for ($n = 0; $n < $count; $n++) {
                    $array[] = $this->instructions[$index + $n];
                }
                $this->eax = $array;
            },

            OpCode::LOOKUP => function() {
                $symbol = $this->popInstruction();
                $this->eax = $this->lookupSymbol($symbol);
                if ($this->eax instanceof TrackedSymbol) {
                    $this->eax = $this->eax->getValue();
                }
            },

            OpCode::MATCH => function() {
                $this->executeRegexMatch();
            },

            OpCode::MOVA => function() {
                $this->eax = $this->popInstruction();
            },

            OpCode::MUL => function() {
                $this->eax *= $this->popStack();
            },

            OpCode::NOT => function() {
                $this->eax = +!$this->eax;
            },

            OpCode::NOP => static function() {},

            OpCode::POPA => function() {
                $length = $this->popInstruction();
                $this->eax = [];
                while ($length--) {
                    array_unshift($this->eax, $this->popStack());
                }
            },

            OpCode::POPSYM => function() {
                $this->executePopSymbol();
            },
            OpCode::POPVAR => function() {
                $this->executePopVar();
            },
            OpCode::POPVARL => function() {
                $this->executePopVarLocal();
            },

            OpCode::POW => function() {
                $base = $this->popStack();
                $this->eax = $base ** $this->eax;
            },

            OpCode::PR0GRAM_INIT => function() {
                $this->eax = 0xCCCCCCCC;
                $this->scope = [];
                $this->stack = [];
            },

            OpCode::PR0GRAM_END => function() use(&$done) {
                $done = true;
                $this->eip = count($this->instructions);
            },

            OpCode::POP => function() {
                $this->eax = $this->popStack();
            },
            OpCode::PUSH => function() {
                $this->pushStack($this->eax);
            },

            OpCode::RET => function() {
                $this->eip = $this->popStack();
            },

            OpCode::SCOPE_INIT => function() {
                array_unshift($this->scope, ['VARS' => []]);
            },
            OpCode::SCOPE_RET => function() {
                array_shift($this->scope);
            },

            OpCode::SHL => function() {
                $left = $this->eax;
                $right = $this->popStack();
                $this->eax = $left << $right;
            },
            OpCode::SHR => function() {
                $left = $this->eax;
                $right = $this->popStack();
                $this->eax = $left >> $right;
            },
            OpCode::SUB => function() {
                $this->eax = $this->popStack() - $this->eax;
            }
        ];

        foreach ($mnemonicExecutors as $opcode => $code) {
            unset($mnemonicExecutors[$opcode]);
            $mnemonicExecutors[OpCode::get($opcode)] = $code;
        }

        $serializer = new BytecodeSerializer();
        $this->instructions = $serializer->unserialize($bytecode);
        $this->eip = 0;
        $this->stack = [];

        $totalInstructions = count($this->instructions);
        while ($this->eip < $totalInstructions) {
            $instruction = $this->popInstruction();
            $code = $mnemonicExecutors[$instruction] ?? function() use($instruction) {
                --$this->eip;
                $this->dump($instruction);
                throw new RunnerException(sprintf('Unimplemented instruction code "%s" at EIP %d', $instruction, $this->eip));
            };

            $code();
        }

        if ($done) {
            return $this->endProgram();
        }

        throw new SeriousBugException('EIP exceeded number of instructions; report this as a bug.');
    }

    /**
     * @return mixed
     */
    protected function endProgram()
    {
        if (0 !== count($this->stack)) {
            $this->dump();
            trigger_error('Unclean program termination.', E_USER_ERROR);
        }

        return $this->eax;
    }

    /**
     * @return mixed
     */
    protected function peekInstruction()
    {
        return $this->instructions[$this->eip];
    }

    /**
     * @return mixed
     */
    protected function popInstruction()
    {
        return $this->instructions[$this->eip++];
    }

    /**
     * @return mixed
     * @throws RunnerException
     */
    protected function popStack()
    {
        if (!count($this->stack)) {
            throw new RunnerException(sprintf('Stack underflow at EIP %d', $this->eip - 1));
        }

        return array_pop($this->stack);
    }

    /**
     * @param mixed ...$values
     * @return BytecodeRunner
     */
    protected function pushStack(...$values): self
    {
        array_push($this->stack, ...$values);
        return $this;
    }

    /**
     * @param string $symbol
     * @return mixed
     * @throws RunnerException
     */
    protected function lookupSymbol(string $symbol)
    {
        foreach ($this->scope as $scope) {
            if (array_key_exists($symbol, $scope['VARS'])) {
                return $scope['VARS'][$symbol];
            }
        }

        return $this->lookupInProvider($symbol);
    }

    /**
     * @param string $symbol
     * @return mixed
     * @throws RunnerException
     */
    protected function lookupInProvider(string $symbol)
    {
        try {
            return $this->symbolProvider->lookup($symbol);
        } catch (UnknownSymbolException $exception) {
            throw new RunnerException($exception->getMessage(), $exception->getCode(), $exception);
        }
    }

    /**
     * @param $frames
     * @return BytecodeRunner
     */
    protected function jump($frames): self
    {
        $this->eip += $frames;
        return $this;
    }

    /**
     * @return BytecodeRunner
     */
    protected function executeCall(): self
    {
        $target = $this->popInstruction();
        $this->pushStack($this->eip);
        $this->eip = $target;

        return $this;
    }

    /**
     * @return BytecodeRunner
     * @throws RunnerException
     */
    protected function executeDot(): self
    {
        $member = $this->eax;
        $value = $this->popStack();

        if (null === $value) {
            $this->eax = null;
            return $this;
        }

        if (is_array($value)) {
            return $this->executeDotArray($value, $member);
        }

        if (is_object($value)) {
            return $this->executeDotObject($value, $member);
        }

        throw new RunnerException(sprintf('Cannot dot-access this: %s', print_r($value, true)), 1591965580);
    }

    /**
     * @param array $value
     * @param $member
     * @return $this
     */
    protected function executeDotArray(array $value, $member): self
    {
        if (is_int($member) && $member < 0) {
            $member += count($value);
        }

        if (!array_key_exists($member, $value)) {
            trigger_error(sprintf('Undefined index "%s"', $member), E_USER_WARNING);
            $this->eax = null;
            return $this;
        }

        $this->eax = $value[$member];
        return $this;
    }

    /**
     * @return BytecodeRunner
     * @throws RunnerException
     */
    protected function executeDotCall(): self
    {
        $parameters = [];
        $numParameters = $p = $this->popStack();
        while ($p--) {
            array_unshift($parameters, $this->popStack());
        }
        $value = $this->popStack();
        $member = $this->eax;

        if (is_object($value)) {
            return $this->executeDotCallObject($value, $member, $numParameters, $parameters);
        }

        if (is_array($value)) {
            return $this->executeDotCallArray($value, $member, $numParameters, $parameters);
        }

        throw new RunnerException(sprintf('Attempted to execute dot-call on incompatible scalar ("%s")', gettype($value)), 1586173617);
    }

    /**
     * @param array $value
     * @param $member
     * @param int $numParameters
     * @param array $parameters
     * @return $this
     * @throws RunnerException
     */
    protected function executeDotCallArray(array $value, $member, int $numParameters, array $parameters): self
    {
        if (!array_key_exists($member, $value)) {
            throw new RunnerException(sprintf('Cannot dot-call undefined member "%s" from array', $member), 1586250225);
        }

        $function = $value[$member];

        if (!is_callable($function)) {
            throw new RunnerException(sprintf('Failed to dot-call non-callable member "%s" from array', $member), 1586250299);
        }

        try {
            $this->eax = $function(...$parameters);
        } catch (Throwable $throwable) {
            throw new RunnerException(sprintf('While dot-calling member "%s" from array: %s', $member, $throwable->getMessage()), 1586250429, $throwable);
        }
        return $this;
    }

    /**
     * @param $value
     * @param $member
     * @param int $numParameters
     * @param array $parameters
     * @return $this
     * @throws RunnerException
     */
    protected function executeDotCallObject($value, $member, int $numParameters, array $parameters): self
    {
        try {
            $reflection = new ReflectionObject($value);
            $method = $reflection->getMethod($member);
        } catch(ReflectionException $exception) {
            throw new RunnerException(sprintf('While attempting to call %s::%s(): %s: %s',
                get_class($value), $member, get_class($exception), $exception->getMessage()), 1541335718);
        }

        try {
            if (!$method->isPublic()) {
                throw new RunnerException('attempted to access non-public method', 1541331233);
            }

            $requiredParameters = $method->getNumberOfRequiredParameters();
            if ($numParameters < $requiredParameters) {
                throw new RunnerException(sprintf('method requires at least %d parameter(s); got %d instead', $requiredParameters, $numParameters), 1541335871);
            }

            $requiredParameters = $method->getNumberOfParameters();
            if ($numParameters > $requiredParameters) {
                throw new RunnerException(sprintf('method accepts %d parameters at most; got %d instead', $requiredParameters, $numParameters), 1541335971); // time code not a coincidence
            }

            $this->eax = $method->invokeArgs($value, $parameters);
            return $this;

        } catch (RunnerException $exception) {
            throw new RunnerException(sprintf('While attempting to call %s::%s(): %s: %s',
                $reflection->getName(), $method->getName(), get_class($exception), $exception->getMessage()),
                $exception->getCode(), $exception);
        }
    }

    /**
     * @param $value
     * @param $member
     * @return $this
     */
    protected function executeDotObject($value, $member): self
    {
        if (property_exists($value, $member)) {
            $this->eax = $value->$member;
            return $this;
        }

        trigger_error(sprintf('Undefined property %s#%s', get_class($value), $member), E_USER_WARNING);
        $this->eax = null;
        return $this;
    }

    /**
     * @return BytecodeRunner
     * @throws RunnerException
     */
    protected function executeInvoke(): self
    {
        $numParameters = $this->popInstruction();
        $methodName = $this->popInstruction();
        $symbol = $this->lookupInProvider($methodName);

        if (!is_callable($symbol)) {
            throw new RunnerException(sprintf('Symbol "%s" is not callable', $methodName), 1541265104);
        }

        $parameters = [];
        while ($numParameters--) {
            array_unshift($parameters, $this->popStack());
        }

        $this->eax = $symbol(...$parameters);

        return $this;
    }

    /**
     * @return BytecodeRunner
     * @throws RunnerException
     */
    protected function executePopSymbol(): self
    {
        $symbolName = $this->peekInstruction();

        try {
            $symbol = $this->lookupSymbol($symbolName);

            if (!$symbol instanceof TrackedSymbol) {
                return $this->executePopVar();
            }

            $this->popInstruction();
            $value = $this->popStack();
            $symbol->setValue($value);

            $this->eax = $value;
        } catch (RunnerException $e) {
            return $this->executePopVar();
        }

        return $this;
    }

    /**
     * @return BytecodeRunner
     * @throws RunnerException
     */
    protected function executePopVar(): self
    {
        $symbol = $this->peekInstruction();

        foreach ($this->scope as $n => $scope) {
            if (array_key_exists($symbol, $scope['VARS'])) {
                $this->popInstruction();
                $this->scope[$n]['VARS'][$symbol] = $this->popStack();
                return $this;
            }
        }

        return $this->executePopVarLocal();
    }

    /**
     * This method MUST NOT modify EAX!
     *
     * @return BytecodeRunner
     * @throws RunnerException
     */
    protected function executePopVarLocal(): self
    {
        $symbol = $this->popInstruction();
        $this->scope[0]['VARS'][$symbol] = $this->popStack();
        return $this;
    }

    /**
     * @return BytecodeRunner
     * @throws RunnerException
     */
    protected function executeRegexMatch(): self
    {
        $pattern = (string) $this->eax;
        $subject = $this->popStack();
        $helper = RegexMatch::getHelper($pattern, $subject);

        if (!$helper->match($matches)) {
            $this->eax = 0;
            return $this;
        }

        $this->eax = 1;

        foreach ($matches as $key => $value) {
            $this->scope[0]['VARS'][$key] = $value;
        }

        return $this;
    }
}
