<?php
/**
 * @author    Oliver Schieche <lispian@schieche.email>
 * @copyright 2018 Oliver Schieche
 */
namespace Ghoti\Tools\Lispian\VM;

use ReflectionClass;

use function array_flip;

/**
 * Class OpCode
 * @package Ghoti\Tools\Lispian\VM
 */
class OpCode
{
    const DUMP              = 0xdead;
    const PR0GRAM_INIT      = 0x4F53;
    const PR0GRAM_END       = 0x4B4B;
    const SCOPE_INIT        = 0x0010;
    const SCOPE_RET         = 0x0011;
    const LOAD              = 0x000d;
    const LOADA             = 0x000e;
    const POPA              = 0x000f;
    const PUSH              = 0x0012;
    const POP               = 0x0014;
    const MOVA              = 0x0015;
    const POPVAR            = 0x0016;
    const POPSYM            = 0x0017;
    const POPVARL           = 0x0018;

    const JMP               = 0x0020;
    const JZ                = 0x0021;
    const JNZ               = 0x0022;
    const CMPEQ             = 0x0023;
    const CMPNE             = 0x0024;
    const CMPLT             = 0x0025;
    const CMPLTE            = 0x0026;
    const CMPGT             = 0x0027;
    const CMPGTE            = 0x0028;
    const CMPIN             = 0x0029;

    const NOT               = 0x0030;
    const CALL              = 0x0031;
    const INVOKE            = 0x0032;
    const RET               = 0x0033;

    const ADD               = 0x0040;
    const SUB               = 0x0041;
    const MUL               = 0x0042;
    const DIV               = 0x0043;
    const POW               = 0x0044;
    const SHL               = 0x0045;
    const SHR               = 0x0046;

    const MATCH             = 0x0055;

    const NOP               = 0x0090; // Gotta follow tradition.
    const LOOKUP            = 0x00F0; // F --> find
    const DOT               = 0x00F1;
    const DOTCALL           = 0x00F2;

    /** @var bool */
    protected static $debug = false;
    /** @var array */
    protected static $debugTokens = [
        self::DUMP          => 'dump',
        self::PR0GRAM_INIT  => 'pr0_init',
        self::PR0GRAM_END   => 'pr0_end',
        self::SCOPE_INIT    => 'stk_init',
        self::SCOPE_RET     => 'stk_pop',
        self::LOAD          => 'load',
        self::LOADA         => 'loada',
        self::PUSH          => 'spush',
        self::POP           => 'spop',
        self::POPA          => 'apop',
        self::MOVA          => 'mova',
        self::POPVAR        => 'vpop',
        self::POPVARL       => 'vpopl',
        self::POPSYM        => 'sympop',
        self::JMP           => 'jmp',
        self::JZ            => 'jz',
        self::JNZ           => 'jnz',
        self::CMPIN         => 'cmp-in',
        self::CMPEQ         => 'cmp-eq',
        self::CMPNE         => 'cmp-ne',
        self::CMPLT         => 'cmp-lt',
        self::CMPLTE        => 'cmp-lte',
        self::CMPGT         => 'cmp-gt',
        self::CMPGTE        => 'cmp-gte',
        self::NOT           => 'not',
        self::CALL          => 'call',
        self::INVOKE        => 'invoke',
        self::RET           => 'ret',
        self::ADD           => 'add',
        self::SUB           => 'sub',
        self::MUL           => 'mul',
        self::DIV           => 'div',
        self::POW           => 'pow',
        self::SHL           => 'shl',
        self::SHR           => 'shr',
        self::MATCH         => 'match',
        self::NOP           => 'nop',
        self::LOOKUP        => 'lookup',
        self::DOT           => 'dot',
        self::DOTCALL       => 'dotcall'
    ];

    /**
     * @param $opcode
     * @return int|string
     */
    public static function get($opcode)
    {
        return self::$debug ? self::$debugTokens[$opcode] : $opcode;
    }

    /**
     * @return array
     */
    public static function getAll(): array
    {
        $class = new ReflectionClass(static::class);
        $opcodes = array_flip($class->getConstants());
        foreach ($opcodes as $opcode => $mnemonic) {
            $opcodes[$opcode] = static::get($opcode);
        }

        return $opcodes;
    }

    /**
     * @param bool $debug
     */
    public static function setDebug(bool $debug)
    {
        self::$debug = $debug;
    }
}
