<?php
/**
 * @author    Oliver Schieche <lispian@schieche.email>
 * @copyright 2018 Oliver Schieche
 */
namespace Ghoti\Tools\Lispian\VM\Compiler;

use Ghoti\Tools\Lispian\Assembly\Contracts\NodeInterface;
use Ghoti\Tools\Lispian\Assembly\Helper\JumpReference;
use Ghoti\Tools\Lispian\Assembly\Node\Statement\Expressions;
use DomainException;

use function array_shift, array_splice, array_unshift, count, sprintf;

/**
 * Class CompileHelper
 *
 * @package Ghoti\Tools\Lispian\VM\Compiler
 */
class CompileHelper
{
    /** @var bool */
    protected $debug = false;
    /** @var JumpReference[] */
    protected $breakTargets = [];
    /** @var StaticData */
    protected $staticData;
    /** @var SymbolTable */
    protected $symbolTable;

    /**
     * CompileHelper constructor.
     *
     * @param StaticData $data
     * @param SymbolTable $table
     */
    public function __construct(StaticData $data, SymbolTable $table)
    {
        $this->staticData = $data;
        $this->symbolTable = $table;
    }

    /**
     * @return bool
     */
    public function isDebug(): bool
    {
        return $this->debug;
    }

    /**
     * @param bool $debug
     * @return CompileHelper
     */
    public function setDebug(bool $debug): CompileHelper
    {
        $this->debug = $debug;
        return $this;
    }

    /**
     * @param array $compiled
     * @param array $code
     * @return CompileHelper
     */
    public function spliceCode(array &$compiled, array $code): self
    {
        array_splice($compiled, count($compiled), 0, $code);

        return $this;
    }

    /**
     * @param array $compiled
     * @param Expressions|null $expressions
     * @return CompileHelper
     */
    public function spliceExpressions(array &$compiled, Expressions $expressions = null): self
    {
        if (null !== $expressions) {
            foreach ($expressions as $node) {
                $this->spliceNode($compiled, $node);
            }
        }

        return $this;
    }

    /**
     * @param array $compiled
     * @param NodeInterface $node
     * @return CompileHelper
     */
    public function spliceNode(array &$compiled, NodeInterface $node): self
    {
        return $this->spliceCode($compiled, $node->compile($this));
    }

    /**
     * @return StaticData
     */
    public function getStaticData(): StaticData
    {
        return $this->staticData;
    }

    /**
     * @return SymbolTable
     */
    public function getSymbolTable(): SymbolTable
    {
        return $this->symbolTable;
    }

    /**
     * Store a temporary jump reference (TJR)
     *
     * TJRs are pushed from while/for/... loops an point to the end of the
     * respective loop. The reference is used while compiling `break` statements
     * in order to get a jump target.
     *
     * @param JumpReference $reference
     * @return CompileHelper
     */
    public function pushJumpReference(JumpReference $reference): self
    {
        array_unshift($this->breakTargets, $reference);
        return $this;
    }

    /**
     * @return JumpReference
     * @throws DomainException
     */
    public function popJumpReference(): JumpReference
    {
        if (0 === count($this->breakTargets)) {
            throw new DomainException('Attempt to pop a jump reference where none is stored');
        }

        return array_shift($this->breakTargets);
    }

    /**
     * @param int $depth
     * @return JumpReference
     * @throws DomainException
     */
    public function getJumpReference(int $depth = 0): JumpReference
    {
        $length = count($this->breakTargets);

        if ($depth >= $length) {
            throw new DomainException(sprintf('Cannot get jump reference %d deep: index exceeded (%d)', $depth, $length));
        }

        return $this->breakTargets[$depth];
    }
}
