<?php
/* Module Parse::Yapphp::Driver
 *
 * This module is part of the Parse::Yapphp package available on your
 * nearest CPAN
 *
 * Any use of this module in a standalone parser make the included
 * text under the same copyright as the Parse::Yapphp module itself.
 *
 * This notice should remain unchanged.
 *
 * Copyright © 1998, 1999, 2000, 2001, Francois Desarmenien.
 * Copyright © 2017 William N. Braswell, Jr.
 * (see the pod text in Parse::Yapphp module for use and distribution rights)
 */
//------------------------------------------------------------------------------
namespace Ghoti\Tools\Lispian\Parser;

abstract class ParserDriver
{
    /** @var array */
    protected $RULES;
    /** @var array */
    protected $STATES;
    /** @var string */
    protected $VERSION = '1.22';

    /** @var mixed */
    protected $CHECK, $DEBUG, $DOTPOS, $ERRST, $NBERR, $STACK, $TOKEN, $VALUE;

    /**
     * @var LexerInterface
     */
    protected $lexer;

    /**
     * @return array
     */
    abstract protected function getRules(): array;

    /**
     * @return array
     */
    abstract protected function getStates(): array;

    /**
     * Driver constructor.
     *
     * @param LexerInterface $lexer
     */
    public function __construct(LexerInterface $lexer)
    {
        $this->setLexer($lexer);
        $this->DEBUG = 0;
        $this->RULES = $this->getRules();
        $this->STATES = $this->getStates();
    }

    /**
     * @return LexerInterface
     */
    public function getLexer(): LexerInterface
    {
        return $this->lexer;
    }

    /**
     * @param LexerInterface $lexer
     * @return self
     */
    public function setLexer(LexerInterface $lexer): self
    {
        $this->lexer = $lexer;
        return $this;
    }

    /**
     * @return null
     */
    public function YYAbort()
    {
        $this->CHECK = 'ABORT';
        return null;
    }

    /**
     * @return null
     */
    public function YYAccept()
    {
        $this->CHECK = 'ACCEPT';
        return null;
    }

    /**
     * @param int|null $debug
     * @return int
     */
    public function YYDebug(int $debug = null)
    {
        if (null !== $debug) {
            $this->DEBUG = $debug;
        }

        return $this->DEBUG;
    }

    /**
     * @return null
     */
    public function YYErrok()
    {
        $this->ERRST = 0;
        return null;
    }

    /**
     * @return null
     */
    public function YYError()
    {
        $this->CHECK = 'ERROR';
        return null;
    }

    /**
     * @return bool
     */
    public function YYNberr()
    {
        return $this->NBERR;
    }

    /**
     * @return bool
     */
    public function YYRecovering(): bool
    {
        return 0 !== $this->ERRST;
    }

    /**
     * @return mixed
     */
    public function YYParse()
    {
        return $this->parse();
    }

    /**
     * @return mixed
     */
    public function YYToken()
    {
        return $this->TOKEN;
    }

    /**
     * @return mixed
     */
    public function YYValue()
    {
        return $this->VALUE;
    }

    /**
     * @param int $flag
     * @param string $message
     * @param mixed ...$arguments
     * @return self
     */
    protected function debug(int $flag, string $message, ...$arguments): self
    {
        if (0 !== ($this->DEBUG & $flag)) {
            $output = \vsprintf($message, $arguments);
            $this->debugOutput($flag, $output);
        }

        return $this;
    }

    /**
     * @param int $flag
     * @param string $output
     */
    protected function debugOutput(int $flag, string $output)
    {
        if ($flag) {
            \fwrite(\STDERR, $output);
        }
    }

    /**
     * @return mixed
     */
    protected function parse()
    {
        $rules = $this->RULES;
        $states = $this->STATES;
        $lexer = $this->getLexer();
        $dbgerror = 0;

        $this->ERRST = 0;
        $this->NBERR = 0;
        $this->TOKEN = null;
        $this->VALUE = null;
        $this->STACK = [[0, null]];
        $this->CHECK = '';

        while (true) {
            $stateno = $this->STACK[\count($this->STACK) - 1][0];
            $actions = $states[$stateno];
            $act = $actions['DEFAULT'] ?? null;

            $this->debug(2, "In state %d:\n", $stateno)
                ->debug(8, "Stack:[%s]\n", \implode(',', \array_map(function($s) {
                    return $s[0];
                }, $this->STACK)));

            if (\array_key_exists('ACTIONS', $actions)) {
                if (null === $this->TOKEN) {
                    list($this->TOKEN, $this->VALUE) = $lexer->lex($this);
                    $this->debug(1, "Needed token; got >%s<.\n", $this->TOKEN);
                }

                if (\array_key_exists($this->TOKEN, $actions['ACTIONS'])) {
                    $act = $actions['ACTIONS'][$this->TOKEN];
                }
            }

            if (null !== $act) {
                if ($act > 0) { // Shift
                    $this->debug(4, "Shift and go to state %d.\n", $act);

                    if ($this->ERRST && 0 === --$this->ERRST && $dbgerror) {
                        $this->debug(16, "**End of error recovery.\n");
                        $dbgerror = 0;
                    }

                    $this->STACK[] = [$act, $this->VALUE];

                    if ('' !== $this->TOKEN) {
                        $this->TOKEN = $this->VALUE = null;
                    }
                    continue;
                }

                // Reduce
                list($lhs,$len,$code) = $rules[-$act];

                if ($act) {
                    $this->debug(4, 'Reduce using rule %d (%d,%d): ', -$act, $lhs, $len);
                } else {
                    $this->YYAccept();
                }

                $this->DOTPOS = $len;

                if ('@' === $lhs[0]) { // In-line rule
                    if (!\preg_match('~^@\d+-(\d+)$~', $lhs, $match)) {
                        throw new \RuntimeException("In-line rule '$lhs' ill-formed; report this as a BUG.");
                    }

                    $this->DOTPOS = (int) $match[1];
                }

                if (!$this->DOTPOS) {
                    $sempar = [];
                } else {
                    $sempar = \array_map(function($s) {
                        return $s[1];
                    }, \array_slice($this->STACK, -$this->DOTPOS));
                }

                if (null === $code) {
                    $semval = $sempar[0] ?? null;
                } else {
                    \array_unshift($sempar, null);
                    $semval = $code($sempar);
                }

                \array_splice($this->STACK, -$len, $len);

                if ('ACCEPT' === $this->CHECK) {
                    $this->debug(4, "Accept.\n");
                    return $semval;
                }

                if ('ABORT' === $this->CHECK) {
                    $this->debug(4, "Abort.\n");
                    return null;
                }

                $stackTop = $this->STACK[\count($this->STACK) - 1];
                $this->debug(4, 'Back to state %d, then ', $stackTop[0]);

                if ('ERROR' !== $this->CHECK) {
                    $this->debug(4, "go to state %d.\n", $states[$stackTop[0]]['GOTOS'][$lhs]);

                    if ($dbgerror && 0 === $this->ERRST) {
                        $this->debug(16, "**End of error recovery.\n");
                        $dbgerror = 0;
                    }

                    $this->STACK[] = [$states[$stackTop[0]]['GOTOS'][$lhs], $semval];
                    $this->CHECK = '';
                    continue;
                }

                $this->debug(4, "Forced error recovery.\n");
                $this->CHECK = '';
            }

            if (!$this->ERRST) {
                $this->ERRST = 1;
                $this->_Error();

                if (!$this->ERRST) { // if 0, then YYErrok has been called
                    continue;        // so continue parsing
                }

                $this->debug(16, "**Entering error recovery.\n");
                ++$dbgerror;
                ++$this->NBERR;
            }

            if (3 === $this->ERRST) { // The next token is invalid: discard it
                if ('' === $this->TOKEN) { // End of input... No hope
                    $this->debug(16, "**At EOF: aborting.\n");
                    return null;
                }

                $this->debug(16, "**Discard invalid token >%s<.\n", $this->TOKEN);
                $this->TOKEN = $this->VALUE = null;
            }

            $this->ERRST = 3;

            while (\count($this->STACK)) {
                $stackTop = $this->STACK[\count($this->STACK) - 1];

                if (!\array_key_exists('ACTIONS', $states[$stackTop[0]]) ||
                    !\array_key_exists('error', $states[$stackTop[0]]['ACTIONS']) ||
                    $states[$stackTop[0]]['ACTIONS']['error'] <= 0) {
                    $this->debug(16, "**Pop state %d.\n", $stackTop[0]);
                    \array_pop($this->STACK);
                } else {
                    break;
                }
            }

            if (0 === \count($this->STACK)) {
                $this->debug(16, "**No state left on stack: aborting.\n");
                return null;
            }

            // Shift the error token
            $stackTop = $this->STACK[\count($this->STACK) - 1];
            $this->debug(16, "**Shift \$error token and go to state %d.\n", $states[$stackTop[0]]['ACTIONS']['error']);
            $this->STACK[] = [$states[$stackTop[0]]['ACTIONS']['error'], null];
        }

        return null;
    }

    /**
     * ...
     */
    protected function _Error()
    {
        print "Parse error.\n";
    }
}
