<?php
/**
 * @author    Oliver Schieche <lispian@schieche.email>
 * @copyright 2018 Oliver Schieche
 */
namespace Ghoti\Tools\Lispian\Assembly\Node;

use Ghoti\Tools\Lispian\Assembly\Contracts\NodeInterface;
use Ghoti\Tools\Lispian\Assembly\Helper\JumpReference;
use Ghoti\Tools\Lispian\Assembly\Node\Instruction\Nop;
use Ghoti\Tools\Lispian\Exception\VM\SeriousBugException;
use Ghoti\Tools\Lispian\Exception\VM\UnresolvableJumpReference;
use Ghoti\Tools\Lispian\VM\BytecodeSerializer;
use Ghoti\Tools\Lispian\VM\Compiler\CompileHelper;
use Ghoti\Tools\Lispian\VM\Compiler\StaticData;
use Ghoti\Tools\Lispian\VM\Compiler\SymbolTable;
use Ghoti\Tools\Lispian\VM\Contracts\SymbolInterface;
use Ghoti\Tools\Lispian\VM\OpCode;
use Ghoti\Tools\Lispian\VM\Symbol\CallableFunction;

use const E_USER_WARNING;
use function array_key_exists, array_map, array_merge, array_reverse, array_splice, assert, count,
    implode, is_array, is_bool, is_float, is_int, is_string, json_encode, printf, sprintf, trigger_error;

/**
 * Class Pr0gram
 * @package Ghoti\Tools\Lispian\Assembly\Node
 */
class Pr0gram extends AbstractNode
{
    /** @var int Number of instructions before data section starts */
    const PROLOG = 3;

    /** @var array */
    protected $resolved;

    /**
     * Pr0gram constructor.
     *
     * @param NodeInterface $root
     */
    public function __construct(NodeInterface $root)
    {
        parent::__construct('pr0', $root);
    }

    /**
     * @param bool $debug
     * @return string
     * @throws SeriousBugException
     */
    final public function toByteCode(bool $debug = false): string
    {
        $staticData = new StaticData();
        $symbolTable = new SymbolTable();
        $serializer = new BytecodeSerializer();
        $compilerHelper = new CompileHelper($staticData, $symbolTable);
        $compilerHelper->setDebug($debug);

        $compiled = $this->compile($compilerHelper);

        return $serializer->serialize($compiled);
    }

    /**
     * @param CompileHelper $helper
     * @return array
     * @throws SeriousBugException
     */
    public function compile(CompileHelper $helper): array
    {
        $compiled = array_merge(
            [OpCode::get(OpCode::PR0GRAM_INIT), 0xdeadbeef],
            $helper->isDebug() ? [OpCode::get(OpCode::DUMP)] : [],
            $this->left->compile($helper),
            [OpCode::get(OpCode::PR0GRAM_END)]
        );

        $dataSection = 1;
        $staticData = $helper->getStaticData()->getStaticData();

        if (!count($staticData)) {
            array_splice($compiled, $dataSection, 1);
        } else {
            array_splice($compiled, $dataSection, 0, [
                OpCode::get(OpCode::JMP), count($staticData)
            ]);
            array_splice($compiled, $dataSection + 2, 1, $staticData);
        }

        $this->resolved = [];
        $compiled = $this->resolveSymbolReferences($helper, $compiled);
        $compiled = $this->resolveJumpReferences($compiled);

        return $this->validateCode($helper, $compiled);
    }

    /**
     * @return bool
     */
    public function isEmpty(): bool
    {
        /** @var AbstractNode $firstNode */
        $firstNode = $this->left;

        return null === $firstNode
            || $firstNode->left[0] instanceof Nop;
    }

    /**
     * @param CompileHelper $helper
     * @param array $code
     * @return array
     */
    protected function resolveSymbolReferences(CompileHelper $helper, array $code): array
    {
        $foundSymbols = [];

        foreach ($code as $index => $instruction) {
            if ($instruction instanceof SymbolInterface) {
                $foundSymbols[$index] = $instruction;
            }
        }

        if (!count($foundSymbols)) {
            return $code;
        }

        /** @var CallableFunction $symbol */
        foreach ($foundSymbols as $index => $symbol) {
            if (array_key_exists($symbol->getName(), $this->resolved)) {
                $code[$index] = $this->resolved[$symbol->getName()];
                continue;
            }

            $injected = [
                OpCode::get(OpCode::SCOPE_INIT),
                /* At this point, EIP is at the top of the stack and
                 * points to the location to jump back to. Pop it and save
                 * it in EAX until all the variables are popped from the stack
                 */
                OpCode::get(OpCode::POP)
            ];

            foreach (array_reverse($symbol->getParameters()) as $parameter) {
                $injected[] = OpCode::get(OpCode::POPVARL);
                $injected[] = $parameter;
            }

            $injected[] = OpCode::get(OpCode::PUSH);
            /** @noinspection SlowArrayOperationsInLoopInspection */
            $injected = array_merge($injected, $symbol->getValue());
            $injected[] = OpCode::get(OpCode::SCOPE_RET);
            $injected[] = OpCode::get(OpCode::RET);

            $code[$index] = count($code);
            $this->resolved[$symbol->getName()] = $code[$index];
            $helper->spliceCode($code, $injected);
        }

        return $this->resolveSymbolReferences($helper, $code);
    }

    /**
     * @param array $code
     * @return array
     * @throws UnresolvableJumpReference
     */
    protected function resolveJumpReferences(array $code): array
    {
        $backReferences = [];
        $knownReferences = [];

        /** @noinspection CallableInLoopTerminationConditionInspection */
        /** @noinspection ForeachInvariantsInspection */
        for ($index = 0; $index < count($code); $index++) {
            if (!$code[$index] instanceof JumpReference) {
                continue;
            }

            /** @var JumpReference $reference */
            $reference = $code[$index];
            $referenceKey = $reference->getKey();

            if ($reference->isTarget()) {
                assert(!array_key_exists($referenceKey, $knownReferences));
                array_splice($code, $index, 1);
                $knownReferences[$referenceKey] = $index--;
                continue;
            }

            if (!array_key_exists($referenceKey, $backReferences)) {
                $backReferences[$referenceKey] = [];
            }

            $backReferences[$referenceKey][] = $index;
        }

        foreach ($backReferences as $key => $indices) {
            if (!array_key_exists($key, $knownReferences)) {
                throw new UnresolvableJumpReference($key);
            }

            $targetIndex = $knownReferences[$key];
            foreach ($indices as $index) {
                $offset = $targetIndex - $index - 1;
                $code[$index] = $offset;
            }

            unset($backReferences[$key]);
        }

        return $code;
    }

    /**
     * @param CompileHelper $helper
     * @param array $compiled
     * @return array
     * @throws SeriousBugException
     */
    protected function validateCode(CompileHelper $helper, array $compiled): array
    {
        foreach ($compiled as $index => $instruction) {
            if (!is_string($instruction) && !is_int($instruction) && !is_float($instruction) && !is_bool($instruction)) {
                $this->dump($compiled, false);
                throw new SeriousBugException(sprintf('Compiled code validation failed at index %d', $index));
            }
        }

        $symbols = $helper->getSymbolTable();
        $unused = [];

        /** @var SymbolInterface $symbol */
        foreach ($symbols as $symbol) {
            if (!$symbol->isUsed()) {
                $unused[] = $symbol->getName();
            }
        }

        if (!empty($unused)) {
            trigger_error(sprintf('Unused symbol(s): %s', implode(', ', array_map(static function($s) {
                return "\"$s\"";
            }, $unused))), E_USER_WARNING);
        }

        return $compiled;
    }

    /**
     * @param array $compiled
     * @param bool $die
     */
    private function dump(array $compiled, bool $die = true)
    {
        foreach ($compiled as $eip => $insn) {
            printf("%2d %s\n", $eip, is_array($insn) ? json_encode($insn) : $insn);
        }

        if ($die) {
            die;
        }
    }
}
