<?php
/**
 * @author    Oliver Schieche <lispian@schieche.email>
 * @copyright 2018 Oliver Schieche
 */
namespace Ghoti\Tools\Lispian\Assembly\Node;

use Ghoti\Tools\Lispian\Assembly\Contracts\NodeInterface;

use function str_repeat;

/**
 * Class AbstractNode
 * @package Ghoti\Tools\Lispian\Assembly\AbstractNode
 */
abstract class AbstractNode implements NodeInterface
{
    /** @var string */
    protected $operator;
    /** @var NodeInterface */
    protected $left;
    /** @var NodeInterface */
    protected $right;
    /** @var NodeInterface */
    protected $next;

    /**
     * AbstractNode constructor.
     *
     * @param string $operator
     * @param NodeInterface|mixed $left
     * @param NodeInterface|mixed $right
     * @param NodeInterface|mixed $next
     */
    public function __construct(string $operator, $left = null, $right = null, $next = null)
    {
        $this->setLeft($left)
            ->setRight($right)
            ->setNext($next)
            ->setOperator($operator);
    }

    /**
     * @return string
     */
    public function __toString(): string
    {
        static $indentLevel = 0;

        $info = $this->stringify();
        if ('' !== $info) {
            $info = " = '$info'";
        }

        $indent = str_repeat('  ', $indentLevel);
        $stringified = "${indent}[" . $this->operator . "]$info\n";

        ++$indentLevel;
        if ($this->left instanceof NodeInterface) {
            $stringified .= $this->left;
        }
        if ($this->right instanceof NodeInterface) {
            $stringified .= $this->right;
        }
        if ($this->next instanceof NodeInterface) {
            $stringified .= $this->next;
        }
        --$indentLevel;

        return $stringified;
    }

    /**
     * @return string
     */
    public function stringify(): string
    {
        return '';
    }

    /**
     * @return string
     */
    public function getOperator(): string
    {
        return $this->operator;
    }

    /**
     * @param string $operator
     * @return AbstractNode
     */
    public function setOperator(string $operator): NodeInterface
    {
        $this->operator = $operator;
        return $this;
    }


    /**
     * @param NodeInterface|mixed $left
     * @return NodeInterface
     */
    public function setLeft($left): NodeInterface
    {
        $this->left = $left;
        return $this;
    }

    /**
     * @param NodeInterface|mixed $right
     * @return NodeInterface
     */
    public function setRight($right): NodeInterface
    {
        $this->right = $right;
        return $this;
    }

    /**
     * @param NodeInterface|mixed $next
     * @return NodeInterface
     */
    public function setNext($next): NodeInterface
    {
        $this->next = $next;
        return $this;
    }
}
