<?php
/**
 * @author    Oliver Schieche <lispian@schieche.email>
 * @copyright 2018 Oliver Schieche
 */
namespace Ghoti\Tools\Lispian\VM\Compiler;

use Ghoti\Tools\Lispian\Assembly\Contracts\NodeInterface;
use Ghoti\Tools\Lispian\Assembly\Node\Statement\Expressions;
use Ghoti\Tools\Lispian\VM\OpCode;

/**
 * Class CompileHelper
 *
 * @package Ghoti\Tools\Lispian\VM\Compiler
 */
class CompileHelper
{
    /** @var StaticData */
    protected $staticData;
    /** @var SymbolTable */
    protected $symbolTable;
    /** @var int */
    protected $codeAlignment = 0;
    /** @var bool */
    protected $debug = false;

    /**
     * CompileHelper constructor.
     *
     * @param StaticData $data
     * @param SymbolTable $table
     */
    public function __construct(StaticData $data, SymbolTable $table)
    {
        $this->staticData = $data;
        $this->symbolTable = $table;
    }

    /**
     * @return int
     */
    public function getCodeAlignment(): int
    {
        return $this->codeAlignment;
    }

    /**
     * @param int $codeAlignment
     * @return CompileHelper
     */
    public function setCodeAlignment(int $codeAlignment): CompileHelper
    {
        $this->codeAlignment = $codeAlignment;
        return $this;
    }

    /**
     * @return bool
     */
    public function isDebug(): bool
    {
        return $this->debug;
    }

    /**
     * @param bool $debug
     * @return CompileHelper
     */
    public function setDebug(bool $debug): CompileHelper
    {
        $this->debug = $debug;
        return $this;
    }

    /**
     * @param array $compiled
     * @param array $code
     * @return CompileHelper
     */
    public function spliceCode(array &$compiled, array $code): self
    {
        \array_splice($compiled, \count($compiled), 0, $code);

        if ($this->codeAlignment) {
            while (0 !== (\count($compiled) % $this->codeAlignment)) {
                $compiled[] = OpCode::get(OpCode::NOP);
            }
        }

        return $this;
    }

    /**
     * @param array $compiled
     * @param Expressions|null $expressions
     * @return CompileHelper
     */
    public function spliceExpressions(array &$compiled, Expressions $expressions = null): self
    {
        if (null !== $expressions) {
            foreach ($expressions as $node) {
                $this->spliceNode($compiled, $node);
            }
        }

        return $this;
    }

    /**
     * @param array $compiled
     * @param NodeInterface $node
     * @return CompileHelper
     */
    public function spliceNode(array &$compiled, NodeInterface $node): self
    {
        return $this->spliceCode($compiled, $node->compile($this));
    }

    /**
     * @return StaticData
     */
    public function getStaticData(): StaticData
    {
        return $this->staticData;
    }

    /**
     * @return SymbolTable
     */
    public function getSymbolTable(): SymbolTable
    {
        return $this->symbolTable;
    }
}
