<?php
/**
 * @author    Oliver Schieche <lispian@schieche.email>
 * @copyright 2018 Oliver Schieche
 */
namespace Ghoti\Tools\Lispian\Assembly\Node\Functions;

use Ghoti\Tools\Lispian\Assembly\Contracts\NodeInterface;
use Ghoti\Tools\Lispian\Assembly\Node\AbstractNode;
use Ghoti\Tools\Lispian\Assembly\Node\Symbolic\Identifier;
use Ghoti\Tools\Lispian\Exception\VM\FunctionParameterException;
use Ghoti\Tools\Lispian\Exception\VM\SeriousBugException;
use Ghoti\Tools\Lispian\VM\Compiler\CompileHelper;
use Ghoti\Tools\Lispian\VM\OpCode;
use Ghoti\Tools\Lispian\VM\Symbol\CallableFunction;

/**
 * Class Call
 * @package Ghoti\Tools\Lispian\Assembly\Node\Functions
 */
class Call extends AbstractNode
{
    /**
     * Call constructor.
     *
     * @param NodeInterface $head
     * @param NodeInterface $parameters
     */
    public function __construct(NodeInterface $head, NodeInterface $parameters)
    {
        parent::__construct('func', $head, $parameters);
    }

    /**
     * @param CompileHelper $helper
     * @return array
     * @throws SeriousBugException
     * @throws FunctionParameterException
     */
    public function compile(CompileHelper $helper): array
    {
        $code = [];
        $numParameters = 0;

        /** @var Identifier $identifier */
        $identifier = $this->left;
        $identifier = $identifier->left;

        /** @var NodeInterface $node */
        foreach ($this->right as $node) {
            ++$numParameters;
            $helper->spliceNode($code, $node);
            $code[] = OpCode::get(OpCode::PUSH);
        }

        if (null === ($symbol = $helper->getSymbolTable()->lookup($identifier))) {
            $code[] = OpCode::get(OpCode::INVOKE);
            $code[] = $numParameters;
            $code[] = $identifier;
        } elseif (!$symbol instanceof CallableFunction) {
            throw new SeriousBugException(\sprintf('Symbol "%s" is not a callable: %s', $symbol->getName(), \get_class($symbol)));
        } else {
            $numRequiredParameters = $symbol->getNumRequiredParameters();

            if ($numParameters > $numRequiredParameters) {
                throw new FunctionParameterException(\sprintf('Too many arguments provided to function "%s": got %d, only need %d', // got d. lol.
                    $symbol->getName(), $numParameters, $numRequiredParameters
                ));
            }

            if ($numParameters < $numRequiredParameters) {
                throw new FunctionParameterException(\sprintf('Too few arguments provided to function "%s": got %d, require %d',
                    $symbol->getName(), $numParameters, $numRequiredParameters
                ));
            }

            // Increase usage counter of symbol
            $symbol->reference();

            $code[] = OpCode::get(OpCode::CALL);
            $code[] = $symbol;
        }

        return $code;
    }
}
