# Lispian documentation #

## Syntax ##

*Lispian* in its syntax is supposed to be similar to [Lisp][wiki-lisp] 
without ever attempting to mimic it. It is therefore ill-advised to 
make attempts in studying Lisp in order to try to understand *Lispian*.

While *Lispian* uses Lisp's distinctive parentheses and prefix notation, 
and borrows some of its keywords, you'll find that some of its linguistic features 
like [symbol distinction][docs-variables] stem from the author's express lack of will 
and/or interest to implement a more close-to-Lisp solution.

For brevity's sake, this documentation will make excessive use of the _imaginary_ PHP
function `lispianRun($source)` which compiles `$source` and executes its byte-code, then
returns the expression's value.

### A "program" ###

A *Lispian* program is a sequence of *statements*. It MAY include comments. Whitespace serves
no other purpose as it separates tokens; it is not otherwise interpreted.

A statement is introduced with an opening parenthesis `(` (0x28) and 
terminated with a closing parenthesis `)` (0x29). The opening token MUST be succeeded by
an _operator_, a _keyword_, a _function name_ or an immediate closing parenthesis in case of e.g.
empty parameter lists in function declarations.

_Keywords_ introduce control-flow statements or operators; _function names_ indicate the beginning of
a function call (built-in or user-defined).

As opposed to "modern" languages, arithmetic (and boolean) expressions also follow the principle
of "operator first, then arguments"; this is called _prefix notation_ and covered in the next
chapter.

### Prefix notation ###

*Lispian* does not linguistically distinguish unary from plurary operations. All expressions
using operators follow prefix notation: operator comes first, then its argument(s).

```lisp
# $x === 4
(eq $x 4)

# $a || !$b
(or $a (not $b))

# ($a > 4) && ($a < 10)
(and (gt $a 4) (lt $a 10))

# 1 + 2 * 3
(+ 1 (* 2 3))

# 1 * 2 + 3
(+ (* 1 2) 3)
```

Note that there is **no operator precedence**: dot does **not** come before dash in either Lisp or
*Lispian*. The last example demonstrates this.

### Comments ###

In order to comment (out) code, *Lispian* knows end-of-line comments. Once the lexical 
scanner encounters a pound sign `#` (0x23), it gobbles the input until a normal newline `\n` (0x0a)
or "special" newline sequence `\r\n` (0x0d, 0x0a) is encountered.

There are **no** block comments like PHP's `/* ... */`.

```lisp
# I am a comment describing the program
(print "Hello world")  # I describe what this line does.
```

### Expression values ###

A core concept of the underlying VM is that _each expression has a value_. How ever small your
*Lispian* code is, `lispianRun()` will always return that expression's value:

```php
$result = lispianRun('(do 1)');
// --> $result === 1

$result = lispianRun('(do "1")');
// --> $result === "1"

$result = lispianRun('(let $x 1337)');
// --> $result === 1337

$result = lispianRun('(eq 1 1)');
// --> $result === 1

$result = lispianRun('(ne 2 2)');
// --> $result === 0

$result = lispianRun('(eq 3 "3")');
// --> $result === 0
```

### Examples ###

For (possibly) more examples, please take a look at the `scripts/` directory in this
library's repository.

#### Hello world ####

```lisp
# Demonstrates Turing completeness
(print "Hello, world.\n")
```

#### Clock ####

```lisp
# Assign sprintf() result to variable $x
(let $now (sprintf "%Y-%m-%d %H:%M:%S"))
# Output via formatted string
(printf "Clock says it's %s.\n" $now)
```

#### Factorials ####

```lisp
# Defines "fac" recursive function
(defun fac ($n) 
    (if (lte $n 0) 
        1 
        (* $n (fac (- $n 1)))
    )
)

# Assign arbitrary value as function argument
(let $x 5)
# Assign function result to $result variable
(let $result (fac $x))
# Print result
(printf "%d! = %d\n" $x $result)
```

[wiki-lisp]: https://en.wikipedia.org/wiki/Lisp_(programming_language) "Lisp (programming language)"
[docs-variables]: ref/variables.md
