#!/usr/bin/env php
<?php
/**
 * @author    Oliver Schieche <lispian@schieche.email>
 * @copyright 2018 Oliver Schieche
 */
namespace Ghoti\Tools\Lispian\Binary;

(function() {
    $possibleAutoloaders = \array_filter([
        \realpath(\dirname(__DIR__) . '/vendor/autoload.php'),
        \realpath(__DIR__ . '/vendor/autoload.php'),
        \realpath(\dirname(__DIR__, 3) . '/autoload.php'),
    ]);

    $autoload = \array_shift($possibleAutoloaders);
    /** @noinspection PhpIncludeInspection */
    require $autoload;
})();

use Ghoti\Tools\Common\Console\Ansi;
use Ghoti\Tools\Common\Console\CommandLine\Reader;
use Ghoti\Tools\Lispian\Assembly\Node\Pr0gram;
use Ghoti\Tools\Lispian\Exception\Exception;
use Ghoti\Tools\Lispian\Parser;

/** @noinspection AutoloadingIssuesInspection */

class LispianCompiler extends Reader
{
    /** @var array */
    protected $options;

    /**
     * LispianCompiler constructor.
     */
    public function __construct()
    {
        parent::__construct('php-lispian compiler', 'Compile Lispian scripts to byte-code');
    }

    /**
     * @return array
     */
    public function getOptions(): array
    {
        return parent::getOptions() + [
            'd|debug=i'         => ['Set debugging level to ARG', 'default' => 0],
            0xf0 => null,
            'f|file=s'          => ['Read script from file ARG', 'required' => true],
            'o|output=s'        => ['Write byte-code to file ARG; if none given, syntax check only', 'default' => null]
        ];
    }

    /**
     * @throws Exception
     */
    public function run()
    {
        $this->options = $this->readOptions();

        if (!\file_exists($this->options['file'])) {
            throw new Exception(\sprintf('No such file: %s', $this->options['file']));
        }

        if (false === ($source = \file_get_contents($this->options['file']))) {
            throw new Exception(\sprintf('Failed to open "%s" for reading.', $this->options['file']));
        }

        $parser = new Parser($source);
        $parser->YYDebug($this->options['debug']);
        /** @var Pr0gram $node */
        $node = $parser->YYParse();

        if (null !== $this->options['output']) {

            if ('<stdout>' === $this->options['output']) {
                $fp = \STDOUT;
            } elseif (false === ($fp = \fopen($this->options['output'], 'wb'))) {
                throw new Exception(\sprintf('Compilation went okay, but failed to open "%s" for writing.', $this->options['output']));
            }

            \fwrite($fp, $node->toByteCode());

            if (\STDOUT !== $fp) {
                \fclose($fp);
            } else {
                print "\n";
            }
        }
    }
}

try {
    $compiler = new LispianCompiler();
    $compiler->run();
} catch (\Throwable $t) {
    print Ansi::C("@{red Error caught}: %s\n\n", \get_class($t));
    print Ansi::C("@{red @{b %s}} at file %s, line %d.\n\n", $t->getMessage(), $t->getFile(), $t->getLine());
    print $t->getTraceAsString() . "\n";
}
