<?php
/**
 * @author    Oliver Schieche <lispian@schieche.email>
 * @copyright 2018 Oliver Schieche
 */
namespace Ghoti\Tools\Lispian\Assembly\Node;

use Ghoti\Tools\Lispian\Assembly\Contracts\NodeInterface;
use Ghoti\Tools\Lispian\Assembly\Helper\JumpReference;
use Ghoti\Tools\Lispian\Exception\VM\SeriousBugException;
use Ghoti\Tools\Lispian\VM\BytecodeSerializer;
use Ghoti\Tools\Lispian\VM\Compiler\ScopeHelper;
use Ghoti\Tools\Lispian\VM\Compiler\StaticData;
use Ghoti\Tools\Lispian\VM\Compiler\SymbolTable;
use Ghoti\Tools\Lispian\VM\Contracts\SymbolInterface;
use Ghoti\Tools\Lispian\VM\OpCode;
use Ghoti\Tools\Lispian\VM\Symbol\CallableFunction;

/**
 * Class Pr0gram
 * @package Ghoti\Tools\Lispian\Assembly\Node
 */
class Pr0gram extends AbstractNode
{
    /** @var int Number of instructions before data section starts */
    const PR0LOG = 3;

    /** @var array */
    protected $resolved;

    /**
     * Pr0gram constructor.
     *
     * @param NodeInterface $root
     */
    public function __construct(NodeInterface $root)
    {
        parent::__construct('pr0', $root);
    }

    /**
     * @return string
     * @throws SeriousBugException
     */
    final public function toByteCode(): string
    {
        $staticData = new StaticData();
        $symbolTable = new SymbolTable();
        $serializer = new BytecodeSerializer();
        $scopeHelper = new ScopeHelper($staticData, $symbolTable);
        $compiled = $this->compile($scopeHelper);

        return $serializer->serialize($compiled);
    }

    /**
     * @param ScopeHelper $helper
     * @return array
     * @throws SeriousBugException
     */
    public function compile(ScopeHelper $helper): array
    {
        $compiled = \array_merge(
            [OpCode::PR0GRAM_INIT, 0xdeadbeef],
            $this->left->compile($helper),
            [OpCode::PR0GRAM_END]
        );

        $staticData = $helper->getStaticData()->getStaticData();

        if (!\count($staticData)) {
            \array_splice($compiled, 1, 1);
        } else {
            \array_splice($compiled, 1, 0, [
                OpCode::JMP, \count($staticData)
            ]);
            \array_splice($compiled, 3, 1, $staticData);
        }

        // TODO: Implement unused symbols warning
        $this->resolved = [];
        $compiled = $this->resolveSymbolReferences($compiled);
        $compiled = $this->resolveJumpReferences($compiled);

        return $compiled;
    }

    /**
     * @param array $code
     * @return array
     */
    protected function resolveSymbolReferences(array $code): array
    {
        $foundSymbols = [];

        foreach ($code as $index => $instruction) {
            if ($instruction instanceof SymbolInterface) {
                $foundSymbols[$index] = $instruction;
            }
        }

        if (!\count($foundSymbols)) {
            return $code;
        }

        /** @var CallableFunction $symbol */
        foreach ($foundSymbols as $index => $symbol) {
            if (\array_key_exists($symbol->getName(), $this->resolved)) {
                $code[$index] = $this->resolved[$symbol->getName()];
                continue;
            }

            $injected = [
                OpCode::SCOPE_INIT,
                /* At this point, EIP is at the top of the stack and
                 * points to the location to jump back to. Pop it and save
                 * it in EAX until all the variables are popped from the stack
                 */
                OpCode::POP
            ];

            foreach (\array_reverse($symbol->getParameters()) as $parameter) {
                $injected[] = OpCode::POPVAR;
                $injected[] = $parameter;
            }

            $injected[] = OpCode::PUSH;
            /** @noinspection SlowArrayOperationsInLoopInspection */
            $injected = \array_merge($injected, $symbol->getValue());
            $injected[] = OpCode::SCOPE_RET;
            $injected[] = OpCode::RET;

            $code[$index] = \count($code);
            $this->resolved[$symbol->getName()] = $code[$index];
            \array_splice($code, \count($code), 0, $injected);
        }

        return $this->resolveSymbolReferences($code);
    }

    /**
     * @param array $code
     * @return array
     * @throws SeriousBugException
     */
    protected function resolveJumpReferences(array $code): array
    {
        $references = [];

        foreach ($code as $index => $instruction) {
            if (!$instruction instanceof JumpReference) {
                continue;
            }

            /** @var JumpReference $reference */
            $reference = $instruction;

            if (!\array_key_exists($reference->getKey(), $references)) {
                if ($reference->isTarget()) {
                    throw new SeriousBugException(\sprintf('Found a new jump reference at EIP %d that is a target', $index), 1541272515);
                }

                $references[$reference->getKey()] = $index;
            } else {
                if (!$reference->isTarget()) {
                    throw new SeriousBugException(\sprintf('Found a known jump reference at EIP %d that is not a target', $index), 1541272717);
                }

                $offset = $index - $references[$reference->getKey()];
                $code[$references[$reference->getKey()]] = $offset + 1;
                $code[$index] = OpCode::NOP;
                unset($references[$reference->getKey()]);
            }
        }

        return $code;
    }
}
