#!env php
<?php
/**
 * @author    Oliver Schieche <lispian@schieche.email>
 * @copyright 2018 Oliver Schieche
 */
namespace Ghoti\Tools\Lispian\Binary;

(function() {
    $possibleAutoloaders = \array_filter([
        \realpath(\dirname(__DIR__) . '/vendor/autoload.php'),
        \realpath(__DIR__ . '/vendor/autoload.php'),
        \realpath(\dirname(__DIR__, 3) . '/autoload.php'),
    ]);

    $autoload = \array_shift($possibleAutoloaders);
    /** @noinspection PhpIncludeInspection */
    require $autoload;
})();

use Ghoti\Tools\Common\Console\Ansi;
use Ghoti\Tools\Common\Console\CommandLine\Reader;
use Ghoti\Tools\Lispian\Assembly\Node\Pr0gram;
use Ghoti\Tools\Lispian\Exception\Exception;
use Ghoti\Tools\Lispian\Parser;
use Ghoti\Tools\Lispian\VM\Runner\BytecodeRunner;
use Ghoti\Tools\Lispian\VM\Support\BuiltinsProvider;

/** @noinspection AutoloadingIssuesInspection */

class LispianRunner extends Reader
{
    /** @var array */
    protected $options;

    /**
     * LispianRunner constructor.
     */
    public function __construct()
    {
        parent::__construct('php-lispian runner', 'Run Lispian scripts from byte-code or source');
    }

    /**
     * @return array
     */
    public function getOptions(): array
    {
        return parent::getOptions() + [
            'd|debug=i'         => ['Set debugging level to ARG', 'default' => 0],
            0xf0 => null,
            'f|file=s'          => ['Read byte-code or script from file ARG', 'required' => true],
            't|type=?'          => ['Set source type to ARG.', 'values' => ['bytecode','script'], 'default' => 'bytecode'],
            0xf1 => null,
            'w|write'           => ['Write script result to STDERR via var_dump', 'default' => false]
        ];
    }

    /**
     * @throws Exception
     */
    public function run()
    {
        $this->options = $this->readOptions();

        if (!\file_exists($this->options['file'])) {
            throw new Exception(\sprintf('No such file: %s', $this->options['file']));
        }

        if (false === ($source = \file_get_contents($this->options['file']))) {
            throw new Exception(\sprintf('Failed to open "%s" for reading.', $this->options['file']));
        }

        if ('script' === $this->options['type']) {
            $parser = new Parser($source);
            $parser->YYDebug($this->options['debug']);
            /** @var Pr0gram $node */
            $node = $parser->YYParse();
            $source = $node->toByteCode();
        }

        $provider = new BuiltinsProvider([
            'argv' => $this->arguments
        ]);
        $runner = new BytecodeRunner($provider);
        $result = $runner->run($source);

        if ($this->options['write']) {
            \ob_start();
            \var_dump($result);
            \fwrite(\STDERR, \ob_get_clean());
        }
    }
}

try {
    $compiler = new LispianRunner();
    $compiler->run();
} catch (\Throwable $t) {
    print Ansi::C("@{red Error caught}: %s\n\n", \get_class($t));
    print Ansi::C("@{red @{b %s}} at file %s, line %d.\n\n", $t->getMessage(), $t->getFile(), $t->getLine());
    print $t->getTraceAsString() . "\n";
}
