<?php
/**
 * @author    Oliver Schieche <spam+php-getopt@oliver-schieche.de>
 */
namespace Ghoti\Common\Utils;

use function count;
use function preg_replace;
use function str_repeat;
use function strlen;
use function strrpos;
use function substr;

/**
 * Class StringWrapper
 * @package Ghoti\Common\Utils
 */
class StringWrapper
{
    /** @var int */
    protected $firstIndention;
    /** @var int */
    protected $leftIndention;
    /** @var int */
    protected $margin;

    /**
     * StringWrapper constructor.
     *
     * @param int $margin Maximum width of wrapped strings.
     * @param int $firstIndention Indention of first line.
     * @param int $leftIndention Indention of remaining lines.
     */
    public function __construct(int $margin, int $firstIndention = 0, int $leftIndention = 0)
    {
        $this->setMargin($margin)
            ->setFirstIndention($firstIndention)
            ->setLeftIndention($leftIndention);
    }

    /**
     * @param string $string
     * @return string[]
     */
    public function __invoke(string $string): array
    {
        return $this->wrap($string);
    }

    /**
     * @return int
     */
    public function getFirstIndention(): int
    {
        return $this->firstIndention;
    }

    /**
     * @param int $firstIndention
     * @return StringWrapper
     */
    public function setFirstIndention(int $firstIndention): StringWrapper
    {
        $this->firstIndention = $firstIndention;
        return $this;
    }

    /**
     * @return int
     */
    public function getLeftIndention(): int
    {
        return $this->leftIndention;
    }

    /**
     * @param int $leftIndention
     * @return StringWrapper
     */
    public function setLeftIndention(int $leftIndention): StringWrapper
    {
        $this->leftIndention = $leftIndention;
        return $this;
    }

    /**
     * @return int
     */
    public function getMargin(): int
    {
        return $this->margin;
    }

    /**
     * @param int $margin
     * @return StringWrapper
     */
    public function setMargin(int $margin): StringWrapper
    {
        $this->margin = $margin;
        return $this;
    }

    /**
     * @param string $input
     * @return string[]
     */
    public function wrap(string $input): array
    {
        $lines = [];
        $string = $this->compactNewLines($input);

        $firstIndention = $this->getFirstIndention();
        $leftIndent = $this->getLeftIndention();
        $wrapAt = $this->getMargin();

        for ($chars = $wrapAt - $firstIndention; strlen($string); $chars = $wrapAt - $leftIndent) {
            $len = strlen($string);

            if ($len < $chars) {
                $lines[] = $string;
                break;
            }

            $pos = strrpos(substr($string, 0, $chars), ' ');

            if (false === $pos) {
                $lines[] = substr($string, 0, $chars);
                $string = (string) substr($string, $chars);
                continue;
            }

            $lines[] = substr($string, 0, $pos);
            $string = (string) substr($string, $pos + 1);
        }

        $indent = str_repeat(' ', $leftIndent);
        for ($n = 1, $l = count($lines); $n < $l; $n++) {
            $lines[$n] = $indent . $lines[$n];
        }

        return $lines;
    }

    /**
     * @param string $string
     * @return string
     */
    protected function compactNewLines(string $string): string
    {
        return (string) preg_replace('~\s*(?:\r?\n)+\s*~', ' ', $string);
    }
}
