<?php
/**
 * @author    Oliver Schieche <php-common@schieche.email>
 * @copyright 2018 Oliver Schieche
 */
namespace Ghoti\Tools\Common\Utils;

use function array_key_exists;
use function array_map;
use function array_splice;
use function count;
use function explode;
use function is_array;
use function is_int;

/**
 * Class ArrayHelper
 * @package Ghoti\Tools\Common\Utils
 */
class ArrayHelper
{
    /**
     * @param array $input
     * @return array
     */
    public static function flatten(array $input): array
    {
        $result = [];

        foreach ($input as $value) {
            if (!is_array($value)) {
                $result[] = $value;
            } else {
                array_splice($result, count($result), 0, static::flatten($value));
            }
        }

        return $result;
    }

    /**
     * @param array $input
     * @param callable $callback
     * @return array
     */
    public static function map(array $input, callable $callback): array
    {
        return array_map($callback, $input);
    }

    /**
     * @param array $source
     * @param array ...$input
     * @return array
     * @noinspection PhpDocSignatureInspection
     */
    public static function merge(array $source, array ...$input): array
    {
        foreach ($input as $item) {
            $source = static::mergeArrays($source, $item);
        }

        return $source;
    }

    /**
     * @param string $delimiter
     * @param string $input
     * @param int|null $limit
     * @return array
     */
    public static function split(string $delimiter, string $input, int $limit = null): array
    {
        if (null !== $limit) {
            return explode($delimiter, $input, $limit);
        }

        return explode($delimiter, $input);
    }
    /**
     * @param array $a
     * @param array $b
     * @param bool $preserveNumericKeys
     * @return array
     */
    private static function mergeArrays(array $a, array $b, bool $preserveNumericKeys = false): array
    {
        foreach ($b as $k => $v) {
            if (array_key_exists($k, $a) || isset($a[$k])) {
                if (!$preserveNumericKeys && is_int($k)) {
                    $a[] = $v;
                } elseif (is_array($v) && is_array($a[$k])) {
                    $a[$k] = static::mergeArrays($a[$k], $v, $preserveNumericKeys);
                } else {
                    $a[$k] = $v;
                }
            } else {
                $a[$k] = $v;
            }
        }

        return $a;
    }
}
