<?php
/**
 * @author    Oliver Schieche <php-common@schieche.email>
 * @copyright 2018 Oliver Schieche
 */
namespace Ghoti\Tools\Common\System;

use Ghoti\Tools\Common\Exception\ExecutionException;

use function array_splice, explode, fclose, fwrite, implode, is_callable, is_resource, proc_close, proc_open,
    stream_get_contents, strlen;

/**
 * Class Process
 * @package Ghoti\Tools\Common\System
 */
class Process
{
    /** @var array */
    protected $arguments;
    /** @var string */
    protected $binary;
    /** @var array */
    protected $environment = [];
    /** @var callable|string */
    protected $input;
    /** @var string */
    protected $workingDirectory;

    /**
     * Process constructor.
     *
     * @param string|null $binary
     * @param mixed ...$arguments
     */
    public function __construct(string $binary = null, ...$arguments)
    {
        if (null !== $binary) {
            $this->setBinary($binary);
        }

        $this->setArguments($arguments)
            ->setWorkingDirectory('/');
    }

    /**
     * @param mixed ...$arguments
     * @return Process
     */
    public function addArguments(...$arguments): self
    {
        array_splice($this->arguments, count($this->arguments), 0, $arguments);
        return $this;
    }
    
    /**
     * @return array
     */
    public function getArguments(): array
    {
        return $this->arguments;
    }

    /**
     * @param array $arguments
     * @return Process
     */
    public function setArguments(array $arguments): self
    {
        $this->arguments = $arguments;
        return $this;
    }

    /**
     * @return string
     */
    public function getBinary(): string
    {
        return $this->binary;
    }

    /**
     * @param string $binary
     * @return Process
     */
    public function setBinary(string $binary): self
    {
        $this->binary = $binary;
        return $this;
    }

    /**
     * @return array
     */
    public function getEnvironment(): array
    {
        return $this->environment;
    }

    /**
     * @param array $environment
     * @return Process
     */
    public function setEnvironment(array $environment): Process
    {
        $this->environment = $environment;
        return $this;
    }

    /**
     * @return bool
     */
    public function hasInput(): bool
    {
        return null !== $this->input;
    }

    /**
     * @return callable|string
     */
    public function getInput()
    {
        return $this->input;
    }

    /**
     * @param callable|string $input
     * @return Process
     */
    public function setInput($input): self
    {
        $this->input = $input;
        return $this;
    }

    /**
     * @return string
     */
    public function getWorkingDirectory(): string
    {
        return $this->workingDirectory;
    }

    /**
     * @param string $workingDirectory
     * @return Process
     */
    public function setWorkingDirectory(string $workingDirectory): self
    {
        $this->workingDirectory = $workingDirectory;
        return $this;
    }

    /**
     * @param array $environment
     * @return array
     * @throws ExecutionException
     */
    public function exec(array $environment = []): array
    {
        [$exitcode, $stdout, $stderr] = $this->execute($environment);

        if ($exitcode) {
            throw new ExecutionException($this, sprintf("Process '%s' failed with code %d: %s\n%s",
                $this->getCommand(), $exitcode, $stderr, $stdout), $exitcode);
        }

        return explode("\n", $stdout);
    }

    /**
     * @param array $environment
     * @return array
     * @throws ExecutionException
     */
    public function execute(array $environment = []): array
    {
        $pipesConfig = [
            1 => ['pipe', 'w'],
            2 => ['pipe', 'w']
        ];

        if ($this->hasInput()) {
            $pipesConfig += [0 => ['pipe', 'r']];
        }

        $handle = proc_open($this->getCommand(), $pipesConfig, $pipes, $this->getWorkingDirectory(), $environment + $this->getEnvironment() + ['LANG' => 'C']);

        if (!is_resource($handle)) {
            throw new ExecutionException($this, sprintf('Failed to fork "%s"', $this->getCommand()));
        }

        if ($this->hasInput()) {
            $input = $this->getInput();

            if (is_callable($input)) {
                $content = $input();
            } else {
                $content = (string) $input;
            }

            fwrite($pipes[0], $content, strlen($content));
            fclose($pipes[0]);
        }

        $stdout = stream_get_contents($pipes[1]);
        $stderr = stream_get_contents($pipes[2]);
        $exitcode = proc_close($handle);

        return [$exitcode, $stdout, $stderr, 'code' => $exitcode, 'stdout' => $stdout, 'stderr' => $stderr];
    }

    /**
     * @return string
     */
    protected function getCommand(): string
    {
        return implode(' ', [
            $this->getBinary(),
            implode(' ', $this->getArguments())
        ]);
    }
}
