<?php
/**
 * @author    Oliver Schieche <php-common@schieche.email>
 * @copyright 2018 Oliver Schieche
 */
namespace Ghoti\Tools\Common\Factory;

use Ghoti\Tools\Common\Configuration\ConfigurationStore;
use Ghoti\Tools\Common\Exception\NotCreatedException;
use Ghoti\Tools\Common\Hydrator\HydratableInterface;
use Ghoti\Tools\Common\Hydrator\HydratorInterface;
use ReflectionClass;
use ReflectionException;
use function array_pop;
use function explode;
use function get_class;
use function implode;
use function property_exists;

/**
 * Class Factory
 * @package Ghoti\Tools\Common\Factory
 */
abstract class Factory implements FactoryInterface
{
    /**
     * @param array $ctorArguments
     * @return object
     * @throws NotCreatedException
     */
    final public static function createObject(...$ctorArguments)
    {
        $config = ConfigurationStore::getConfiguration();
        $factoryConfig = $config[FactoryInterface::class];

        if (!property_exists($factoryConfig, static::class)) {
            throw new NotCreatedException(sprintf('Factory "%s" is not configured.', static::class));
        }

        $className = $factoryConfig->{static::class};

        try {
            $reflection = new ReflectionClass($className);
            return $reflection->newInstanceArgs($ctorArguments);
        } catch(ReflectionException $exception) {
            throw new NotCreatedException(sprintf('Factory "%s" failed to create object of class "%s": %s',
                static::class, $className, $exception->getMessage()), 0, $exception);
        }
    }

    /**
     * @param HydratableInterface $object
     * @return HydratorInterface
     * @throws NotCreatedException
     */
    final public static function createHydrator(HydratableInterface $object): HydratorInterface
    {
        /** @var ReflectionClass[] $knownHydrators */
        static $knownHydrators = [];

        $config = ConfigurationStore::getConfiguration();
        $hydratorConfig = $config[HydratorInterface::class];
        $objectClass = get_class($object);

        if (!isset($knownHydrators[$objectClass])) {
            if (null !== $hydratorConfig && property_exists($hydratorConfig, $objectClass)) {
                $hydratorClass = $hydratorConfig->{$objectClass};
            } else {
                $hydratorClass = explode('\\', $objectClass);
                $className = array_pop($hydratorClass);
                $hydratorClass[] = 'Hydrator';
                $hydratorClass[] = "${className}Hydrator";
                $hydratorClass = implode('\\', $hydratorClass);
            }

            try {
                $knownHydrators[$objectClass] = new ReflectionClass($hydratorClass);
            } catch(ReflectionException $exception) {
                throw new NotCreatedException(sprintf('Factory "%s" failed to create hydrator "%s" for object of class "%s": %s',
                    static::class, $hydratorClass, $objectClass, $exception->getMessage()), 0, $exception);
            }
        }

        $reflection = $knownHydrators[$objectClass];

        /** @var HydratorInterface $hydrator */
        /** @noinspection OneTimeUseVariablesInspection */
        $hydrator = $reflection->newInstance();
        return $hydrator;
    }
}
