<?php
/**
 * @author    Oliver Schieche <php-common@schieche.email>
 * @copyright 2018 Oliver Schieche
 */
namespace Ghoti\Tools\Common\Console\CommandLine;

use Ghoti\Tools\Common\Console\Ansi;
use Ghoti\Tools\Common\Exception\RuntimeException;
use Ghoti\Tools\Common\Utils\StringWrapper;

/**
 * Class Reader
 * @package Ghoti\Tools\Common\Console\CommandLine
 */
class Reader
{
    /** @var array */
    protected $arguments; //!< Program arguments
    /** @var array */
    protected $parsedOptions; //!< Parsed options
    /** @var array */
    protected $longOptions; //!< Long options
    /** @var string */
    protected $programName;
    /** @var string */
    protected $programTitle;
    /** @var string */
    protected $programDescription;
    /** @var array */
    protected $shortOptions; //!< Shorthand options

    /** @var number */
    const USAGE_MARGIN = 80; //!< Maximum wrap-edge for usage summaries
    /** @var number */
    const USAGE_OPTIONS_LENGTH = 29; //!< Characters option summaries are usually long

    /** @var string */
    static protected $ruleParsePattern = '~^
		(?:(?P<negatable>!))?
		(?:(?P<short>[a-z])\|)?
		(?P<long>[a-z](?:[a-z-]*)[a-z])
		(?:
			(?:=(?P<type>[si@?]))
			|
			(?P<multiple>\+)
		)?
			$~xi';

    /**
     * Reader constructor.
     *
     * @param string $programTitle
     * @param string $programDescription
     */
    public function __construct(string $programTitle, string $programDescription)
    {
        $this->parsedOptions = null;

        $this->setProgramName($this->determineProgramName())
            ->setArguments($this->determineProgramArguments())
            ->setProgramTitle($programTitle)
            ->setProgramDescription($programDescription);
    }

    /**
     * @return array
     */
    public function getOptions(): array
    {
        return [
            'h|help' => ['Display this help message and abort the program', [$this, 'showUsage']],
            'q|quiet' => 'Do not display any verbosity messages; warnings and errors only',
            'v|verbose+' => 'Increase verbosity of progress messages',
            0xFFFF => NULL // This is for a new-line after "generic" options
        ];
    }

    /**
     * @param array $arguments
     * @return Reader
     */
    public function addArguments(array $arguments): Reader
    {
        foreach ($arguments as $arg) {
            $this->arguments[] = $arg;
        }
        return $this;
    }

    /**
     * @return array
     */
    public function getArguments(): array
    {
        return $this->arguments;
    }

    /**
     * @param array $arguments
     * @return Reader
     */
    public function setArguments(array $arguments): Reader
    {
        $this->arguments = $arguments;
        return $this;
    }

    /**
     * @return string
     */
    public function getProgramName(): string
    {
        return $this->programName;
    }

    /**
     * @param string $programName
     * @return Reader
     */
    public function setProgramName(string $programName): Reader
    {
        $this->programName = $programName;
        return $this;
    }

    /**
     * @return string
     */
    public function getProgramTitle(): string
    {
        return $this->programTitle;
    }

    /**
     * @param string $programTitle
     * @return Reader
     */
    public function setProgramTitle(string $programTitle): Reader
    {
        $this->programTitle = $programTitle;
        return $this;
    }

    /**
     * @return string
     */
    public function getProgramDescription(): string
    {
        return $this->programDescription;
    }

    /**
     * @param string $programDescription
     * @return Reader
     */
    public function setProgramDescription(string $programDescription): Reader
    {
        $this->programDescription = $programDescription;
        return $this;
    }

    /**
     * @param number $exitCode
     * @param string $message
     * @param array $arguments
     * @throws \Ghoti\Tools\Common\Exception\RuntimeException
     */
    public function showUsage($exitCode, $message = null, ...$arguments)
    {
        $wrapper = new StringWrapper(static::USAGE_MARGIN, 0, 8);
        $optionsWrapper = new StringWrapper(static::USAGE_MARGIN, 2, static::USAGE_OPTIONS_LENGTH);

        if (null !== $message) {
            $errorMessage = \vsprintf("Error: $message", $arguments);

            $color = new Ansi();

            foreach ($wrapper($errorMessage) as $line) {
                \fwrite(\STDERR, $color("@{red %s}\n", $line));
            }

            \fwrite(\STDERR, "\n");
        }

        $caption = \implode(' - ', [$this->getProgramTitle(), $this->getProgramDescription()]);
        foreach ($wrapper($caption) as $line) {
            \fwrite(\STDERR, "$line\n");
        }

        \fwrite(\STDERR, "\n");

        \fprintf(\STDERR, "\nUsage: %s [options...]\n\nValid options:\n\n",
            $this->getProgramName());

        $haveRequired = false;
        $options = $this->getOptions();

        foreach ($options as $ruleDesc => $ruleOptions) {
            if (\is_int($ruleDesc)) {
                if (null === $ruleOptions) {
                    \fprintf(\STDERR, "\n");
                } else if (\is_array($ruleOptions)) {
                    foreach ($wrapper("$ruleOptions[0]:") as $line) {
                        \fprintf(\STDERR, "%s\n", $line);
                    }
                }
                
                continue;
            }

            if (!\preg_match(static::$ruleParsePattern, $ruleDesc, $m)) {
                throw new RuntimeException(sprintf('Invalid rule pattern: "%s"', $ruleDesc));
            }

            $ruleOptions = (array) $ruleOptions;

            $m += ['type' => ''];

            $line = '  ';
            $long = $m['long'];

            if (isset($m['negatable'][0])) {
                $long = "(no-)$long";
            }

            if ('' === $m['short']) {
                $long = "--$long";
            } else {
                $long = " [--$long]";
                $line = "$line-$m[short]";
            }

            $line = "$line$long";

            if (!empty($ruleOptions['required'])) {
                $haveRequired = true;
                $line .= '*';
            }

            if ('' !== $m['type']) {
                $line .= ' ARG';
            }

            if (static::USAGE_OPTIONS_LENGTH > ($l = \strlen($line) + 2)) {
                $line .= \str_repeat(' ', static::USAGE_OPTIONS_LENGTH - $l);
            }

            $line = "$line: ";
            \fwrite(\STDERR, $line);

            /** @noinspection MultiAssignmentUsageInspection */
            list($description) = $ruleOptions;
            $description = \trim(\preg_replace('/\s+/', ' ', $description));

            if (!\preg_match('~[[:punct:]]$~', $description)) {
                $description .= '.';
            }

            $argumentHelp = [];

            if ('?' === $m['type'] && isset($ruleOptions['values']) && 1 !== \count($ruleOptions['values'])) {
                $values = \array_map(function($s) {
                    return '"' . $s . '"';
                }, $ruleOptions['values']);

                $last = \array_pop($values);
                $argumentHelp[] = sprintf('ARG should be one of %s or %s',
                    \implode(', ', $values), $last);
            }

            if ($m['type'] && isset($ruleOptions['default'])) {
                $argumentHelp[] = sprintf('default is "%s"', $ruleOptions['default']);
            }

            if (\count($argumentHelp)) {
                $description .= ' (' . \implode('; ', $argumentHelp) . ')';
            }

            $optionsWrapper->setFirstIndention(\strlen($line));
            foreach ($optionsWrapper($description) as $line) {
                \fwrite(\STDERR, "$line\n");
            }
        }

        \fwrite(\STDERR, "\n");
        if ($haveRequired) {
            fwrite(\STDERR, "Parameters marked with * are required.\n\n");
        }

        exit($exitCode);
    }

    /**
     * @return array
     * @throws \Ghoti\Tools\Common\Exception\RuntimeException
     */
    public function readOptions(): array
    {
        $this->parsedOptions = [];

        $this->parseRuleset($this->getOptions());

        $argumentValues = [];

        while (\count($this->arguments)) {
            $argument = \array_shift($this->arguments);

            /* Anything not beginning with a dash or being a lone dash
             * is considered an argument. Empty arguments are possible.
             */
            if ('' === $argument || '-' === $argument || '-' !== $argument[0]) {
                $argumentValues[] = $argument;
                continue;
            }

            // Abort options processing when encountering a "--"
            if ('--' === $argument) {
                \array_splice($argumentValues, \count($argumentValues), 0, $this->arguments);
                break;
            }

            // Short options processing
            if ('-' !== $argument[1]) {
                $key = $argument[1];
                if (!\array_key_exists($key, $this->shortOptions)) {
                    $this->error('No such option "-%s"', $key);
                }
                $rule = $this->longOptions[$key = $this->shortOptions[$key]];

                if (2 < \strlen($argument)) {
                    if ($rule['type']) {
                        \array_unshift($this->arguments, \substr($argument, 2));
                    } else {
                        \array_unshift($this->arguments, '-' . \substr($argument, 2));
                    }
                }
            } else {
            // Long options processing
                $key = \substr($argument, 2);
                if (false !== \strpos($key, '='))
                {
                    list($key,$value) = \explode('=', $key, 2);
                    \array_unshift($this->arguments, $value);
                }

                if (\array_key_exists($key, $this->longOptions)) {
                    $rule = $this->longOptions[$key];
                } else if (0 !== \strpos($key, 'no-')) {
                    $rule = null; // Triggers "rule is undefined" code inspection later
                    $this->error('No such option "--%s"', $key);
                } else {
                    $key = \substr($key, 3);
                    if (!\array_key_exists($key, $this->longOptions) || !$this->longOptions[$key]['negatable']) {
                        $this->error('No such option "--no-%s" or negatable "--%s"', $key, $key);
                    }
                    $rule = $this->longOptions[$key];
                    $rule['_negate'] = true;
                }
            }

            $value = null;

            if (isset($rule['conflicts'])) {
                $conflicts = [];
                foreach ((array) $rule['conflicts'] as $c) {
                    if (isset($this->parsedOptions[$c])) {
                        $conflicts[] = $c;
                    }
                }
                if (!empty($conflicts)) {
                    $this->showUsage(1, 'Seen option "%s" but that conflicts with option(s) %s', $key, $this->concat($conflicts));
                }
            }

            if ($rule['multiple']) {
                if (!\array_key_exists($key, $this->parsedOptions)) {
                    $this->parsedOptions[$key] = 0;
                }

                ++$this->parsedOptions[$key];
                continue;
            }

            if ($rule['type']) {
                if (null === ($value = $this->getValue())) {
                    $this->error('Argument "--%s" needs a value', $key);
                }

                if (!\array_key_exists($key, $this->parsedOptions) && '@' === $rule['type']) {
                    $this->parsedOptions[$key] = [];
                }

                switch($rule['type'])
                {
                    case '@':
                        $this->parsedOptions[$key][] = $value;
                        continue 2; // Skip everything below
                    case 's':
                        break;
                    case 'i':
                        if (!\is_numeric($value)) {
                            $this->error('Argument "%s" to "--%s" is not numeric.', $value, $key);
                        }
                        break;
                    case '?':
                        if (!$rule['options']['values'] || !\in_array($value, $rule['options']['values'], true)) {
                            $this->error('Value "%s" to argument "--%s" invalid.', $value, $key);
                        }
                        break;
                    default:
                        $this->error('Rule type "%s" unknown', $rule['type']);
                }
            }

            if (isset($rule['options'][1]) && \is_callable($rule['options'][1])) {
                $args = [$value, $key, &$this->parsedOptions];
                $value = \call_user_func_array($rule['options'][1], $args);
            }

            if (null === $value) {
                $value = (isset($rule['_negate']) ? false : true);
            }

            $this->parsedOptions[$key] = $value;
        }

        $this->checkRequired();
        $this->setArguments($argumentValues);

        return $this->getParsedOptions();
    }

    /**
     * @return array
     * @throws \Ghoti\Tools\Common\Exception\RuntimeException
     */
    protected function getParsedOptions(): array
    {
        if (null === $this->parsedOptions) {
            throw new RuntimeException('Options were not yet parsed.');
        }

        return $this->parsedOptions;
    }

    /**
     * @param array $ruleSet
     * @return Reader
     * @throws \Ghoti\Tools\Common\Exception\RuntimeException
     */
    protected function parseRuleset(array $ruleSet): Reader
    {
        $this->longOptions = [];
        $this->shortOptions = [];

        foreach ($ruleSet as $ruleDesc => $ruleOptions) {
            if (\is_int($ruleDesc) && (null === $ruleOptions || \is_array($ruleOptions))) {
                continue;
            }
            
            $ruleOptions = (array) $ruleOptions;

            if (!\preg_match(static::$ruleParsePattern, $ruleDesc, $m)) {
                throw new RuntimeException(sprintf('Invalid rule pattern: "%s"', $ruleDesc));
            }

            $m += ['multiple' => false, 'type' => ''];
            $rule = ['type' => '', 'multiple' => ''];

            if ($m['negatable']) {
                $rule['negatable'] = true;
            }

            if ($m['multiple']) {
                $rule['multiple'] = true;
            } else if ($m['type']) {
                $rule['type'] = $m['type'];
            }

            if (isset($ruleOptions['required'])) {
                if (isset($ruleOptions['default'])) {
                    throw new RuntimeException(sprintf('Option "%s" is required, but provides a default value', $m['long']));
                }

                $rule['required'] = true;
                unset($ruleOptions['required']);
            }

            if (isset($ruleOptions['conflicts'])) {
                if (!\is_array($ruleOptions['conflicts'])) {
                    $ruleOptions['conflicts'] = \explode(',', $ruleOptions['conflicts']);
                }
                $rule['conflicts'] = $ruleOptions['conflicts'];
                unset($ruleOptions['conflicts']);
            }

            if ('' !== $m['short']) {
                if (\array_key_exists($m['short'], $this->shortOptions)) {
                    throw new RuntimeException(sprintf('Option spec "%s" redefines short rule "%s"', $ruleDesc, $m['short']));
                }

                $this->shortOptions[$m['short']] = $m['long'];
            }

            if (\array_key_exists($m['long'], $this->longOptions)) {
                throw new RuntimeException(sprintf('Option spec "%s" redefines long rule "%s"', $ruleDesc, $m['long']));
            }
            $this->longOptions[$m['long']] = $rule + ['options' => $ruleOptions];

            if (isset($ruleOptions['default'])) {
                $this->parsedOptions[$m['long']] = $ruleOptions['default'];
            }
        }

        return $this;
    }

    /**
     * Make sure mandatory parameters are provided; display usage if not
     * @throws \Ghoti\Tools\Common\Exception\RuntimeException
     */
    protected function checkRequired()
    {
        $missing = [];

        foreach ($this->longOptions as $option => $parameters) {
            if (!isset($parameters['required']) || !$parameters['required']) {
                continue;
            }
            if (\array_key_exists($option, $this->parsedOptions)) {
                continue;
            }

            /* Don't complain about unsatisfied obligation if a
             * conflicting parameter is present
             */
            if (isset($parameters['conflicts'])) {
                foreach ((array) $parameters['conflicts'] as $c) {
                    if (\array_key_exists($c, $this->parsedOptions)) {
                        continue 2;
                    }
                }
            }

            $missing[] = $option;
        }

        if (!empty($missing)) {
            $this->showUsage(1, 'The following required %s missing: %s',
                (1 === \count($missing) ? 'parameter is' : 'parameters are'),
                $this->concat($missing));
        }
    }

    /**
     * @return array
     */
    protected function determineProgramArguments(): array
    {
        return \array_slice($_SERVER['argv'], 1);
    }

    /**
     * @return string
     */
    protected function determineProgramName(): string
    {
        return \cli_get_process_title();
    }

    /**
     * @return null|mixed
     */
    private function getValue()
    {
        if (!\count($this->arguments)) {
            return null;
        }

        $value = $this->arguments[0];
        if ('' === $value || '-' === $value || '-' !== $value[0]) {
            return \array_shift($this->arguments);
        }

        if ('--' !== $value) {
            return null;
        }

        \array_shift($this->arguments);
        if (!\count($this->arguments)) {
            return null;
        }
        // Take value regardless of possible prefixes
        $value = \array_shift($this->arguments);
        // Unshift the terminator i.o.t. let this function work again
        \array_unshift($this->arguments, '--');
        return $value;
    }

    /**
     * @param string $format
     * @param array $arguments
     * @throws \Ghoti\Tools\Common\Exception\RuntimeException
     */
    private function error($format, ...$arguments)
    {
        $this->showUsage(1, '%s', \vsprintf($format, $arguments));
    }

    /**
     * @param array $vWords
     * @return string
     */
    private function concat(array $vWords): string
    {
        $last = array_pop($vWords);
        if (empty($vWords)) {
            return "'$last'";
        }

        return sprintf("%s and '%s'", \implode(', ', array_map(function($s) {
            return "'$s'";
        }, $vWords)), $last);
    }
}
