<?php
/**
 * @author    Oliver Schieche <php-common@schieche.email>
 * @copyright 2018 Oliver Schieche
 */
namespace Ghoti\Tools\Common\Console;

use Ghoti\Tools\Common\Exception\InvalidArgumentException;

/**
 * Class Ansi
 * @package Ghoti\Tools\Common\Console
 */
class Ansi
{
    /** @var string */
    const TAG_START = '@{';
    /** @var string */
    const TAG_END = '}';

    /** @var bool */
    static protected $useColors;
    /** @var string */
    protected $format;
    /** @var array */
    protected $arguments;

    /**
     * @param string $format
     * @param array ...$arguments
     * @return Ansi
     */
    public static function C(string $format = null, ...$arguments): Ansi
    {
        $instance = new static();
        if (null !== $format) {
            $instance->setFormat($format)->setArguments($arguments);
        }
        return $instance;
    }

    /**
     * @param bool $enable
     */
    public static function enableColors(bool $enable)
    {
        static::$useColors = $enable;
    }

    /**
     * Ansi constructor.
     *
     * @param string|null $format
     * @param array ...$arguments
     */
    public function __construct(string $format = null, ...$arguments)
    {
        if (null !== $format) {
            $this->setFormat($format)->setArguments($arguments);
        }
    }

    /**
     * @param string $format
     * @param array ...$arguments
     * @return Ansi
     */
    public function __invoke(string $format, ...$arguments)
    {
        return $this->setFormat($format)->setArguments($arguments);
    }

    /**
     * @return string
     */
    public function __toString()
    {
        try {
            return $this->colorize();
        } catch (InvalidArgumentException $exception) {
            return $exception->getMessage();
        }
    }

    /**
     * @return string
     */
    public function getFormat(): string
    {
        return $this->format;
    }

    /**
     * @param string $format
     * @return Ansi
     */
    public function setFormat(string $format): Ansi
    {
        $this->format = $format;
        return $this;
    }

    /**
     * @return array
     */
    public function getArguments(): array
    {
        return $this->arguments;
    }

    /**
     * @param array $arguments
     * @return Ansi
     */
    public function setArguments(array $arguments): Ansi
    {
        $this->arguments = $arguments;
        return $this;
    }

    /**
     * @return string
     * @throws \Ghoti\Tools\Common\Exception\InvalidArgumentException
     */
    public function colorize(): string
    {
        $message = $this->convertTags($this->getFormat());
        $message = \vsprintf($message, $this->getArguments());

        return $message;
    }

    /**
     * @param string $format
     * @param array $inEffect
     * @return string
     * @throws InvalidArgumentException
     */
    protected function convertTags(string $format, array &$inEffect = []): string
    {
        if (false === ($p = \strpos($format, static::TAG_START))) {
            return $format;
        }

        $tl = \strlen(self::TAG_START);
        $start = $p;

        for ($p += $tl, $n = 1, $l = \strlen($format); $n > 0 && $p < $l; $p++) {
            if (self::TAG_START === \substr($format, $p, $tl)) {
                ++$n;
                $p += $tl;
            } else if (self::TAG_END === $format[$p]) {
                --$n;
            }
        }

        if ($n > 0) {
            throw new InvalidArgumentException("Unterminated tags in '$format'");
        }

        return \substr($format, 0, $start)
            . $this->parseTagLine($inEffect, \substr($format, $start + $tl, $p - $start - 1 - $tl))
            . $this->convertTags(\substr($format, $p), $inEffect);
    }

    /**
     * @param array $inEffect
     * @param string $tagLine
     * @return string
     * @throws InvalidArgumentException
     */
    protected function parseTagLine(array &$inEffect, string $tagLine): string
    {
        static $ESC = "\x1B[";

        if (null === static::$useColors) {
            static::initialize();
        }

        $parts = \explode(' ', $tagLine, 2);
        $code = $this->getCode($parts[0]);

        if (1 === \count($parts)) {
            return '';
        }

        if (false === static::$useColors) {
            return $parts[1];
        }

        if (!isset($inEffect[0])) {
            $reset = '';
        } else {
            $reset = $ESC . \implode(';', $inEffect);
        }

        $inEffect[] = $code;
        $embed = $this->convertTags($parts[1], $inEffect);
        $result = $ESC . $code . 'm' . $embed . $reset;
        \array_pop($inEffect);

        if (!isset($inEffect[0])) {
            $result .= $ESC . '0m';
        }

        return $result;
    }

    /**
     * @param string $code
     * @return mixed
     * @throws InvalidArgumentException
     */
    protected function getCode(string $code)
    {
        static $codes = [
            'black'     => 30,
            'red'       => 31,
            'green'     => 32,
            'yellow'    => 33,
            'blue'      => 34,
            'magenta'   => 35,
            'cyan'      => 36,
            'white'     => 37,
            'u'         => 4,
            'b'         => 1
        ];

        if (!\array_key_exists($code, $codes)) {
            throw new InvalidArgumentException("Invalid formatting code '$code'");
        }

        return $codes[$code];
    }

    /**
     * Initially configure instances
     */
    protected static function initialize()
    {
        static::$useColors = true;
    }
}