<?php
/**
 * @author    Oliver Schieche <php-common@schieche.email>
 * @copyright 2018 Oliver Schieche
 */
namespace Ghoti\Tools\Common\Configuration;

use Ghoti\Tools\Common\Utils\ArrayHelper;

/**
 * Class ConfigurationStore
 * @package Ghoti\Tools\Common\Configuration
 */
class ConfigurationStore implements \ArrayAccess
{
    /** @var ConfigurationStore */
    static protected $store;
    /** @var array */
    protected $configurationValues;

    /**
     * @return ConfigurationStore
     */
    public static function getConfiguration(): ConfigurationStore
    {
        if (null === static::$store) {
            static::$store = new static();
        }

        return static::$store;
    }

    /**
     * @param ConfigurationInterface $configuration
     * @return ConfigurationStore
     */
    public function merge(ConfigurationInterface $configuration): ConfigurationStore
    {
        $this->configurationValues = ArrayHelper::merge(
            $this->configurationValues, $configuration->getConfiguration());
        return $this;
    }

    /**
     * Whether a offset exists
     * @link http://php.net/manual/en/arrayaccess.offsetexists.php
     * @param mixed $offset <p>
     * An offset to check for.
     * </p>
     * @return boolean true on success or false on failure.
     * </p>
     * <p>
     * The return value will be casted to boolean if non-boolean was returned.
     * @since 5.0.0
     *
     */
    public function offsetExists($offset): bool
    {
        return \array_key_exists($offset, $this->configurationValues);
    }

    /**
     * Offset to retrieve
     * @link http://php.net/manual/en/arrayaccess.offsetget.php
     * @param mixed $offset <p>
     * The offset to retrieve.
     * </p>
     * @return mixed Can return all value types.
     * @since 5.0.0
     */
    public function offsetGet($offset)
    {
        return $this->configurationValues[$offset];
    }

    /**
     * Offset to set
     * @link http://php.net/manual/en/arrayaccess.offsetset.php
     * @param mixed $offset <p>
     * The offset to assign the value to.
     * </p>
     * @param mixed $value <p>
     * The value to set.
     * </p>
     * @return void
     * @since 5.0.0
     */
    public function offsetSet($offset, $value)
    {
        $this->configurationValues[$offset] = $value;
    }

    /**
     * Offset to unset
     * @link http://php.net/manual/en/arrayaccess.offsetunset.php
     * @param mixed $offset <p>
     * The offset to unset.
     * </p>
     * @return void
     * @since 5.0.0
     */
    public function offsetUnset($offset)
    {
        unset($this->configurationValues[$offset]);
    }

    /**
     * ConfigurationStore constructor.
     */
    protected function __construct()
    {
        $this->configurationValues = [];
    }
}
