<?php
/**
 * @author    Oliver Schieche <github+atlassian-client@spam.oliver-schieche.de>
 * @copyright 2019-2021
 */

namespace Ghoti\Test\AtlassianClient\Jira\Client;

use Ghoti\AtlassianClient\Exception\Http\BadPayloadException;
use Ghoti\AtlassianClient\Exception\Http\NotFoundException;
use Ghoti\AtlassianClient\Exception\Jira\IssueNotFoundException;
use Ghoti\AtlassianClient\Exception\Jira\UnrecoverableException;
use Ghoti\AtlassianClient\Jira\Client\JiraClient;
use Ghoti\AtlassianClient\Jira\Issue\JiraIssue;
use GuzzleHttp\Psr7\Response;
use PHPUnit\Framework\TestCase;

/**
 * Class JiraClientTest
 *
 * @package Ghoti\Test\AtlassianClient\Jira\Client
 */
class JiraClientTest extends TestCase
{
    /**
     * Test client should succeed.
     */
    public function test_Client_getIssueByKey_shouldSucceed(): void
    {
        $mb = $this->getMockBuilder(JiraClient::class)
                   ->onlyMethods(['get']);
        $mock = $mb->getMock();

        $mock->expects($this->once())
            ->method('get')
            ->willReturn([
                'id' => '1337',
                'key' => 'ABC-1337',
                'self' => 'https://jira.example.com/browse/ABC-1337',
                'fields' => []
            ]);
        $issue = $mock->getIssueByKey('ABC-1337');

        $this->assertInstanceOf(JiraIssue::class, $issue);
        $this->assertEquals('1337', $issue->getId());
        $this->assertEquals('ABC-1337', $issue->getKey());
    }

    /**
     * Test client fails with invalid JSON payload
     */
    public function test_Client_getIssueByKey_failIssueNotFound(): void
    {
        $mb = $this->getMockBuilder(JiraClient::class)
                   ->onlyMethods(['call']);
        $mock = $mb->getMock();

        $mock->expects($this->once())
             ->method('call')
             ->willThrowException(new NotFoundException('SPLURT-9000'));

        $this->expectException(IssueNotFoundException::class);
        $this->expectExceptionMessage('SPLURT-9000');
        $mock->getIssueByKey('SPLURT-9000');
    }

    /**
     * Test client fails with invalid JSON payload
     */
    public function test_Client_getIssueByKey_failInvalidContentType(): void
    {
        $mb = $this->getMockBuilder(JiraClient::class)
                   ->onlyMethods(['call']);
        $mock = $mb->getMock();

        $mock->expects($this->once())
             ->method('call')
             ->willReturn(new Response(200, ['Content-Type' => 'text/plain; charset=UTF-8'], 'No content here.'));

        $this->expectException(UnrecoverableException::class);
        $this->expectExceptionMessage("Result of call to '/rest/api/2/issue/GORK-1234' is not JSON");
        $mock->getIssueByKey('GORK-1234');
    }

    /**
     * Test client fails with invalid JSON payload
     */
    public function test_Client_getIssueByKey_failInvalidJSON(): void
    {
        $mb = $this->getMockBuilder(JiraClient::class)
                   ->onlyMethods(['call']);
        $mock = $mb->getMock();

        $mock->expects($this->once())
             ->method('call')
             ->willReturn(new Response(200, ['Content-Type' => 'application/json; charset=UTF-8'], 'Invalid JSON'));

        $this->expectException(UnrecoverableException::class);
        $this->expectExceptionMessage('Response has unparsable JSON:');
        $mock->getIssueByKey('GORK-1111');
    }
}
