<?php
declare(strict_types=1);
/**
 * @author    Oliver Schieche <github+atlassian-client@spam.oliver-schieche.de>
 * @copyright 2019-2021
 */
namespace Ghoti\AtlassianClient\Jira\Issue;

use ArrayAccess;
use Ghoti\AtlassianClient\Jira\Issue\Field\JiraIssueField;
use Ghoti\AtlassianClient\Jira\Issue\Field\JiraIssueFieldFactory;

use const E_USER_ERROR;
use function array_key_exists, sprintf, trigger_error;

/**
 * Class JiraIssue
 *
 * @package Ghoti\AtlassianClient\Jira\Client\Issue
 * @implements ArrayAccess<string,mixed>
 */
class JiraIssue implements ArrayAccess
{
    /** @var JiraIssueField[] */
    protected $fields = [];
    /** @var int */
    protected $id;
    /** @var string */
    protected $key;
    /** @var string */
    protected $url;

    /**
     * @param string $key
     * @return JiraIssueField|null
     */
    public function getField(string $key): ?JiraIssueField
    {
        return $this->fields[$key] ?? null;
    }

    /**
     * @param string $key
     * @param mixed $value
     * @return $this
     */
    public function setField(string $key, $value): self
    {
        if (!array_key_exists($key, $this->fields)) {
            $this->fields[$key] = JiraIssueFieldFactory::create($key, $value);
        } else {
            $this->fields[$key]->setValue($value);
        }

        return $this;
    }

    /**
     * @return int
     */
    public function getId(): int
    {
        return $this->id;
    }

    /**
     * @param int $id
     * @return JiraIssue
     */
    public function setId(int $id): self
    {
        $this->id = $id;
        return $this;
    }

    /**
     * @return string
     */
    public function getKey(): string
    {
        return $this->key;
    }

    /**
     * @param string $key
     * @return JiraIssue
     */
    public function setKey(string $key): self
    {
        $this->key = $key;
        return $this;
    }

    /**
     * @return string
     */
    public function getUrl(): string
    {
        return $this->url;
    }

    /**
     * @param string $url
     * @return JiraIssue
     */
    public function setUrl(string $url): self
    {
        $this->url = $url;
        return $this;
    }

    /**
     * @return int
     */
    public function getStatus(): int
    {
        return (int) $this['status']['id'];
    }

    /**
     * @param mixed $offset
     * @return bool
     */
    public function offsetExists($offset)
    {
        return array_key_exists($offset, $this->fields);
    }

    /**
     * @param mixed $offset
     * @return mixed|void
     */
    public function offsetGet($offset)
    {
        return $this->fields[$offset]->getValue();
    }

    /**
     * @param mixed $offset
     * @param mixed $value
     */
    public function offsetSet($offset, $value)
    {
        $this->fields[$offset]->setValue($value);
    }

    /**
     * @param mixed $offset
     */
    public function offsetUnset($offset)
    {
        trigger_error(sprintf('%s::%s is not allowed', static::class, __METHOD__), E_USER_ERROR);
    }
}
