<?php
declare(strict_types=1);
/**
 * @author    Oliver Schieche <github+atlassian-client@spam.oliver-schieche.de>
 * @copyright 2019-2021
 */
namespace Ghoti\AtlassianClient\Jira\Client;

use Ghoti\AtlassianClient\Exception\Http\BadPayloadException;
use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\ClientException;
use Ghoti\AtlassianClient\Contracts\ExceptionInterface;
use Ghoti\AtlassianClient\Exception\Http\BadRequestException;
use Ghoti\AtlassianClient\Exception\Http\NotFoundException;
use Ghoti\AtlassianClient\Exception\Http\TransferException;
use Ghoti\AtlassianClient\Exception\Jira\IssueNotFoundException;
use Ghoti\AtlassianClient\Exception\Jira\UnrecoverableException;
use Ghoti\AtlassianClient\Jira\Issue\JiraIssue;

use GuzzleHttp\Exception\GuzzleException;
use JsonException;
use Psr\Http\Message\ResponseInterface;
use function implode;
use function json_decode;
use Ghoti\AtlassianClient\Jira\Issue\JiraIssueFactory;
use function sprintf;
use function strpos;
use const JSON_THROW_ON_ERROR;

/**
 * Class JiraClient
 *
 * @package Ghoti\AtlassianClient\Jira\Client
 */
class JiraClient
{
    /** @var ClientInterface */
    protected $client;

    /**
     * JiraClient constructor.
     *
     * @param ClientInterface|null $client
     */
    public function __construct(ClientInterface $client = null)
    {
        if (null !== $client) {
            $this->setClient($client);
        }
    }

    /**
     * @return ClientInterface
     */
    public function getClient(): ClientInterface
    {
        return $this->client;
    }

    /**
     * @param ClientInterface $client
     * @return JiraClient
     */
    public function setClient(ClientInterface $client): JiraClient
    {
        $this->client = $client;
        return $this;
    }

    /**
     * @param string $issueKey
     * @return JiraIssue
     * @throws IssueNotFoundException
     * @throws UnrecoverableException
     */
    public function getIssueByKey(string $issueKey): JiraIssue
    {
        try {
            $data = $this->get("/rest/api/2/issue/$issueKey");
            return JiraIssueFactory::create($data);
        } catch (NotFoundException $exception) {
            throw new IssueNotFoundException($issueKey, 0, $exception);
        } catch (ExceptionInterface $exception) {
            throw new UnrecoverableException($exception->getMessage(), $exception->getCode(), $exception);
        }
    }

    /**
     * @param string $method
     * @param string $uri
     * @return ResponseInterface
     * @throws BadRequestException
     * @throws NotFoundException
     * @throws TransferException
     */
    protected function call(string $method, string $uri): ResponseInterface
    {
        try {
            $response = $this->client->request($method, $uri);
        } catch (ClientException $exception) {
            switch ($exception->getCode()) {
                case 400:
                    throw new BadRequestException("Bad request in call to $uri", 400, $exception);
                case 404:
                    throw new NotFoundException($this->extractErrorMessage($exception), 404, $exception);
                default:
                    throw new TransferException($exception->getMessage(), $exception->getCode(), $exception);
            }
        } catch (GuzzleException $exception) {
            throw new TransferException($exception->getMessage(), $exception->getCode(), $exception);
        }

        return $response;
    }

    /**
     * @param string $uri
     * @return array<string,mixed|mixed[]>
     * @throws TransferException
     * @throws NotFoundException
     * @throws BadRequestException
     * @throws BadPayloadException
     */
    protected function get(string $uri): array
    {
        $response = $this->call('get', $uri);
        $headers = $response->getHeader('Content-Type');

        if (empty($headers) || false === strpos($headers[0], 'application/json')) {
            throw new TransferException("Result of call to '$uri' is not JSON");
        }

        try {
            return json_decode((string) $response->getBody(), true, 512, JSON_THROW_ON_ERROR);
        } catch(JsonException $exception) {
            throw new BadPayloadException('Response has unparsable JSON: ' . $exception->getMessage(), $exception->getCode(), $exception);
        }
    }

    /**
     * @param ClientException $exception
     * @return string
     */
    protected function extractErrorMessage(ClientException $exception): string
    {
        if (null === ($response = $exception->getResponse())) {
            return sprintf('No response: %s', $exception->getMessage());
        }

        $contentType = $response->getHeader('Content-Type');

        if (empty($contentType) || false === strpos($contentType[0], 'application/json')) {
            return sprintf('%d %s', $response->getStatusCode(), $response->getReasonPhrase());
        }

        $data = json_decode((string) $response->getBody(), true);
        return implode(', ', $data['errorMessages']);
    }
}
